////////////////////////////////////////////////////////////////


module videomodule ( 
	input clk,		// clock input @25MHz
	input [15:0]d,	// data read from external RAM
	output [15:0]a,	// external RAM address
	//input [7:0]pald,	// data for palette RAM
	//input [3:0]pala,	// address for palette RAM
	//input palwr,		// write to palette RAM
	output hsyn,	// Horizontal sync. pulse (active low)
	output vsyn,	// Vertical sync. pulse (active low)
	output [11:0]video,	// Video output (12 bits/pixel)
	output vrd,		// video read
	output hblk,	// Horizontal blank
	output vblk,	// Vertical blank
	output [9:0]hc	// Horizontal counter
);

// Horizontal section
// Total: 800 pixels (32 us)
// Displayed: 512 pixels
// hsync width: 96 pixels
reg [9:0]hc=0;	// Horizontal counter
assign peb=hc[0];
reg hsyn=1;		// sync
reg hblk=0;		// blank output (active if pixel >= 512)

always @(posedge clk) begin
	if (hc==10'd799) begin hc<=0; hblk<=0; end else hc<=hc+1;
	if (hc==10'd591) hsyn<=0;
	if (hc==10'd687) hsyn<=1;
	if (hc==10'd511) hblk<=1;
end

// Vertical section
// Total: 525 lines
// Displayed: 480 lines
// vsync width: 2 lines

reg [9:0]vc=0;	// Vertical Counter
reg vsyn=1;		// sync.
reg vblk=0;		// blank output (active if line >=240)

always @(negedge hblk) begin
	if (vc==10'd524) begin vc<=0; vblk<=0; end else vc<=vc+1;
	if (vc==10'd449) vsyn<=0;
	if (vc==10'd451) vsyn<=1;
	if (vc==10'd399) vblk<=1;
end

// Global Blank
wire blk;
assign blk = hblk|vblk;

// Memory Address
assign a = {vc[8:0],hc[8:2]} + 16'h1000;

reg dblk=0;	// blk delayed
always @(posedge clk) dblk<=blk;

// Video read

assign vrd = (~blk) & (~hc[1]) & (~hc[0]);

reg [15:0]vsh;

always @(posedge clk) vsh <= vrd ? d : {vsh[11:0],4'bxxxx};

wire [3:0]pixel;

assign pixel = dblk ? 4'b0000 : vsh[15:12];

assign video[11]=pixel[2];
assign video[10]=pixel[3];
assign video[ 9]=pixel[2];
assign video[ 8]=pixel[3];

assign video[ 7]=pixel[1];
assign video[ 6]=pixel[3];
assign video[ 5]=pixel[1];
assign video[ 4]=pixel[3];

assign video[ 3]=pixel[0];
assign video[ 2]=pixel[3];
assign video[ 1]=pixel[0];
assign video[ 0]=pixel[3];


/*
// Palette RAM (two-port, 32-byte, RAM)
genram2p #(
	.INITFILE("pal.hex"),
	.AW(5),
	.DW(8)
	) pal0 (.clk(clk),.ao({dblk,pixel}),.we(palwr),.data_in(pald),.ai({1'b0,pala}),.data_out(video));

*/
endmodule

//----------------------------------------------------------------------------
//-- Generic two-port RAM
//----------------------------------------------------------------------------

module genram2p (        
    input clk,                      
    input wire [AW-1: 0] ao,      //-- Address for output port (reads)
    input wire [AW-1: 0] ai,      //-- Address for input port (writes)
    input wire we,                  //-- Write Enable
    input wire [DW-1: 0] data_in,   //-- Input data
    output reg [DW-1: 0] data_out   //-- Output data
);
//-- Parameters
parameter INITFILE = "pal.hex";	// Initial caontent
parameter AW = 4;		// Address width (bits)
parameter DW = 8;		// Data width (bits)

  //-- Memory
  reg [DW-1: 0] ram [0: (1<<AW)-1];

  //-- Read and writing
  always @(posedge clk) begin
    data_out <= ram[ao];
    if (we) ram[ai] <= data_in;
  end
  
integer j;
initial 
  for(j = 0; j < (1<<AW); j = j+1) 
    ram[j] = 0;
    
// Initial data from HEX file
//initial begin
//  $readmemh(INITFILE, ram);
//end

endmodule

