//--------------------------------------------------------------------
// GUSY
// by Jesús Arias (2021)
//--------------------------------------------------------------------

module SYSTEM (
	input clk,		// Main clock input 25MHz
	input reset,	// Global reset (active high)

	output xwe,		// external RAM: WE (active low)
	output xoe,		// external RAM: OE (active low)
	output [15:0]xa,// external RAM: address bus
	output [15:0]xdo,	// external RAM: output data
	input  [15:0]xdi,	// external RAM: input data

	output hsyn,	// Horizontal Sync. (active low, every 32us) 
	output vsyn,	// Vertical Sync. (active low, every 525 lines)
	output [11:0]video,	// RGB444 output

	input  [7:0]pinin,	// Input port (includes MISO)
	output [7:0]pinout,	// Output port (includes SCK, MOSI, /SS_flash, and /SS_SD)

	output	pwmout,	// Audio output (1-bit, it can be PWM modulated)

	input	rxd,	// UART
	output 	txd,
	
	output sck,		// SPI
	output mosi,
	input  miso,	
	output fss,		// Flash SS
	output sss		// SD SS

    
    //,output	deb0,
    //output	deb1

);

///////////////////////////////////////////////////////
///////////////// external memory //////////////

wire [15:0]cdo;	// CPU output
wire [15:0]cdi;	// CPU data input
wire [15:0]ca;	// CPU address
wire [15:0]va;	// Video address
wire vrd;		// Video read
wire cclk;		// CPU clock

wire we;		// Write Enable from CPU

assign xoe = ~((~we) | vrd);	// external Output Enable (active low)
assign xwe = ~(we & (~bcs) & (~cclk));	// external Write Enable (active low)

///////////////////////////////////////////////////////
///////////////// internal memory //////////////

wire [15:0]bdo;	// data output
wire bcs;		// boot select ($0000 to $0FFF)
wire bwe;		// boot write enable

assign bcs=(~ca[15])&(~ca[14])&(~ca[13])&(~ca[12]);
assign bwe=we & bcs;

genram #(
    .INITFILE("rom.hex"),   
	.AW(12),
	.DW(16)
	) ram0 (.clk(cclk),.ca(ca[11:0]),.we(bwe),.data_in(cdo),.data_out(bdo));
	
/////////////////////////////////////////////////////////
///////////////////// Peripherals ///////////////////////
/////////////////////////////////////////////////////////

wire iocs;		// IO select ($0020 to $003F)

assign iocs=(ca[15:5]==11'h001);
wire iowe0,iowe1,iowe2,iowe3,iowe4,iowe5,iowe6,iowe7;	// Write strobes
assign iowe0=(we)&(iocs)&(~ca[4])&(~ca[3])&(~ca[2])&(~ca[1])&(~ca[0]);
assign iowe1=(we)&(iocs)&(~ca[4])&(~ca[3])&(~ca[2])&(~ca[1])&( ca[0]);
assign iowe2=(we)&(iocs)&(~ca[4])&(~ca[3])&(~ca[2])&( ca[1])&(~ca[0]);
assign iowe3=(we)&(iocs)&(~ca[4])&(~ca[3])&(~ca[2])&( ca[1])&( ca[0]);
assign iowe4=(we)&(iocs)&(~ca[4])&(~ca[3])&( ca[2])&(~ca[1])&(~ca[0]);
assign iowe5=(we)&(iocs)&(~ca[4])&(~ca[3])&( ca[2])&(~ca[1])&( ca[0]);
assign iowe6=(we)&(iocs)&(~ca[4])&(~ca[3])&( ca[2])&( ca[1])&(~ca[0]);
assign iowe7=(we)&(iocs)&(~ca[4])&(~ca[3])&( ca[2])&( ca[1])&( ca[0]);

wire iore1,iore2;	// Read strobes
assign iore1=(~we)&(iocs)&(~ca[4])&(~ca[3])&(~ca[2])&(~ca[1])&( ca[0]);
assign iore2=(~we)&(iocs)&(~ca[4])&(~ca[3])&(~ca[2])&( ca[1])&(~ca[0]);

/////////////////////////////
// I/O data to CPU MUX
reg [15:0]iodo;
always@*
  case (ca[2:0])
     0 : iodo <=  {10'h000,irqen}; // IRQ Enable
     1 : iodo <=  timerval;        // Timer
     2 : iodo <=  {8'h00,uart_do}; // UART rx
     3 : iodo <=  spi_do;          // SPI RX
     4 : iodo <=  {tirq,7'h0,vblk,hblk,ove,fe,spibusy,tend,thre,dv}; // PFLAGS
     5 : iodo <=  16'hxxxx;
     6 : iodo <=  16'hxxxx;
     7 : iodo <=  16'hxxxx;
     default : iodo <= 16'hxxxx;
  endcase

/////////////////////////////
// UART
wire tend,thre,dv,fe,ove; // Flags
wire [7:0] uart_do;	// output data
UART_CORE #(.DIVISOR(217)) uart0(.txd(txd), .tend(tend), .thre(thre), .d(cdo[7:0]), .wr(iowe2),
	.q(uart_do), .dv(dv), .fe(fe), .ove(ove), .rxd(rxd), .rd(iore2), .clk(clk) );

/////////////////////////////
// SPI
wire [15:0] spi_do;
wire spibusy;
wire [2:0]ssb;

SPI spi0( .clk(clk), .d(cdo), .wr0(iowe3), .wr1(iowe5), .q(spi_do), .busy(spibusy),
		  .SCK(sck), .MOSI(mosi), .MISO(miso), .SSB(ssb) );
assign fss=ssb[0];
assign sss=ssb[1];

/////////////////////////////
// Timer
reg [15:0]timerval=0;	// Counter
reg [15:0]timermax=0;	// Max value reg
reg tirq=0;				// IRQ
wire max;				// max count
assign max = (timerval==timermax);

always @ (posedge clk)
begin
	timermax <= iowe1 ? cdo : timermax;
	timerval <= (iowe1|max) ? 0 : timerval+1;
	tirq  <= max ? 1 : (iore1 ? 0 : tirq);
end

///////////////////////////////////////////////
// PWM (9-bit)
// uses the video horizontal counter (0 to 799)

reg pwmout=0;
reg [8:0]pwmval;

always @ (posedge clk)
begin
	pwmval <= iowe4 ? cdo[8:0] : pwmval;	
	pwmout <= ({1'b0,pwmval}==hc) ? 1'b0  : ( hblk ? 1'b1: pwmout);
end

//////////////////////////////////////
// Video interrupts
// posted on horizontal/vertical blank
// Cleared by writing to address 0x27 (bits 0,1 clear H,V IRQs. No data is stored)

reg ohblk=0;	// for edge detection
reg ovblk=0;
reg hirq=0;		// Horizontal IRQ
reg virq=0;		// Vertical IRQ

always @ (posedge clk)
begin
	ohblk<=hblk;
	ovblk<=vblk;
	hirq<= (hblk & (~ohblk)) ? 1'b1 : ( (iowe7 & cdo[0]) ? 1'b0 : hirq);
	virq<= (vblk & (~ovblk)) ? 1'b1 : ( (iowe7 & cdo[1]) ? 1'b0 : virq);
end

/////////////////////////////
// IRQS
wire irq;
wire [2:0]ivector;
wire [6:0]irqs;

assign irq = irqs[0]|irqs[1]|irqs[2]|irqs[3]|irqs[4]|irqs[5]|irqs[6];

// Priority encoder for IRQ vector generation
assign ivector = (irqs[6]?3'h7:(irqs[5]?3'h6:(irqs[4]?3'h5:(irqs[3]?3'h4:
				 (irqs[2]?3'h3:(irqs[1]?3'h2:(irqs[0]?3'h1:3'bxxx) ))))));

// IRQEN reg
reg [5:0]irqen=0;	
always @(posedge clk or posedge reset ) 
	if (reset) irqen<=6'h0;
	else irqen <= iowe0 ? cdo[5:0] : irqen;


// IRQs posted
assign irqs[6] = 1'b0;
assign irqs[5] = virq & irqen[5];	// Vsync
assign irqs[4] = hirq & irqen[4];	// Hsync
assign irqs[3] = dv   & irqen[3];	// UART RX
assign irqs[2] = thre & irqen[2];	// UART TX av.
assign irqs[1] = tirq & irqen[1];	// Timer
assign irqs[0] = 1'b0; 

////////////////////////////////////////////////////
///////////////////// CLOCKS ///////////////////////

assign cclk = clk | (vrd & (~bcs));

////////////////////////////////////////////////////
//////////////////// VIDEO /////////////////////////

wire hblk,vblk;	// Blank pulses
wire [9:0]hc;	// Horizontal counter

videomodule vmod (
	.clk(clk), .d(xdi), .a(va), //.pald(cdo), .pala(ca[3:0]), .palwr(palwr),
	.hsyn(hsyn), .vsyn(vsyn), .vrd(vrd), .video(video),
	.hblk(hblk), .vblk(vblk), .hc(hc)
);

//////////////////////////////////////////////////
//////					CPU					//////
//////////////////////////////////////////////////

CPU_1CV2 cpu0( .clkin(cclk), .we(we), .a(ca), .Do(cdo), .di(cdi), 
	.waitin(1'b0), .resetb(~reset), .irq(irq), .ivector(ivector) );

//////////////////////////////////////////////////
//////		Address and Data input MUX		//////

assign xa=vrd? va : ca;
assign xdo=cdo;
assign cdi = iocs ? iodo : (bcs ? bdo : xdi);

endmodule

//----------------------------------------------------------------------------
//-- Memoria RAM genérica
//----------------------------------------------------------------------------

module genram (        //-- Puertos
    input clk,                      //-- Señal de reloj global
    input wire [AW-1: 0] ca,      //-- Direcciones
    input wire we,                  //-- Write Enable
    input wire [DW-1: 0] data_in,   //-- Dato de entrada
    output reg [DW-1: 0] data_out   //-- Dato a escribir
);
//-- Parametro: Nombre del fichero con el contenido de la RAM
parameter INITFILE = "rom.hex";
parameter AW = 8;
parameter DW = 16;

//-- Calcular el numero de posiciones totales de memoria
//localparam NPOS = 2 ** AW;
//localparam NPOS = 1<<AW;

  //-- Memoria
  reg [DW-1: 0] ram [0: (1<<AW)-1];

  //-- Lectura de la memoria
  always @(negedge clk) begin
    data_out <= ram[ca];
  end

  //-- Escritura en la memoria
  always @(negedge clk) begin
    if (we)
     ram[ca] <= data_in;
  end

//-- Cargar en la memoria el fichero ROMFILE
//-- Los valores deben estan dados en hexadecimal
initial begin
  $readmemh(INITFILE, ram);
end

endmodule


