;---------------------------------------------------------------------
; Convenciones usadas en este programa:
; - R7 es el puntero de pila.
; - R6 se usa para guardar la dirección de retorno en las subrutinas
; - La pila es del tipo "Full-Descending"
; - R0, R1 y R2 se usan para pasar parámetros y retornar resultados.
;               No se necesitan preservar en las subrutinas
; - R3, R4 y R5 se usan para las variables locales.
;               Su valor se debe preservar en las subrutinas
;---------------------------------------------------------------------
;---------------------------------------------------------------------
;----------- REGISTROS de E/S -----------

IOBASE=		0x20
IRQEN=		0x20
TIMER=		0x21
UARTDAT=	0x22
SPIDAT=		0x23
PWM=		0x24
PFLAGS=		0x24
SPICTRL= 	0x25
RESIRQ=		0x27

; offsets respecto de IOBASE
OIRQEN=		0
OTIMER=		1
OUARTDAT=	2
OSPIDAT=	3
OPWM=		4
OPFLAGS=	4
OSPICTRL= 	5
ORESIRQ=	7

;--------- Habilitación de interrupciones --------
EINTEX=		1		; Habilita IRQ externa
ETIMER=		2		; Habilita IRQ Timer
EUARTTHRE=	4		; Habilita IRQ UART TX
EUARTDV=	8		; Habilita IRQ UART RX
EHBLK=		0x10	; Habilita IRQ Horizontal blanking
EVBLK=		0x20	; Habilita IRQ Vertical blanking

;--------- bits de PFLAGS -----------
UARTDV=		1		; Dato válido en UART RX
UARTTHRE=	2		; Listo para transmitir en UART TX
UARTTEND=	4		; Transmisión completa en UART TX
SPIBUSY=	8		; SPI ocupado transfiriendo datos
UARTFE=		0x10	; Error de trama (bit de STOP en 0) en UART RX
UARTOVE=	0x20	; Error de overrun en UART RX
TIMERIRQ=	0x8000	; Flag de IRQ del temporizador

;--------- bits de RESIRQ -----------

RESHBLK=	1		; Dato para borrar IRQ Horizontal Blanking
RESVBLK=	2		; Dato para borrar IRQ vertical Blanking

;--------------------------------------
;----------- CONSTANTES -----------------
;-------------------------------------- 

X0VAL=	-3700	;-4080
Y0VAL=	2000	; 2400
DELTA=  10

cntl=	64		; direcciones de variables globales
cnth=	65

;--------------------------------------
;----------- VECTORES -----------------
;-------------------------------------- 

		org 0			; RESET
inicio:	jr		restart

		org		0x4		; IRQ 0
		org		0x8		; IRQ Timer
		org		0xc		; IRQ TX
		org		0x10	; IRQ RX
		org		0x14	; IRQ Hblk
		jr		hblkirq
		
		org		0x18	; IRQ Vblk
		org		0x1C	; IRQ 6


		org		0x100
;--------------------------------------
;-------- INTERRUPTS --------------
;-------------------------------------- 

hblkirq:
		subi    r7,r7,2     		; R0, R1 -> stack
        st      (r7),r0
        st      (r7+1),r1

		ldi		r1,IOBASE
		ldi		r0,RESHBLK
		st		(r1+ORESIRQ),r0		; clear flag IRQ (HBlank)

		ld		r0,(r1+OSPIDAT)		; Get sample from SPI
		add		r0,r0,r0			; 2*sample (9-bits)
		st		(r1+OPWM),r0
		st		(r1+OSPIDAT),r0		; request another sample
		
		ldi		r1,cntl				;(cnth:cntl)--
		ld		r0,(r1)
		jnz		hlab1
		subi	r0,r0,1
		st		(r1),r0
		addi	r1,r1,1
		ld		r0,(r1)
		jnz		hlab1
		ldi		r1,IOBASE			; if cnthl==0  IRQ off
		ldi		r0,0
		st		(r1+OIRQEN),r0
		st		(r1+OSPICTRL),r0	; 8 bits, no SSx
		jr		hlab2		
hlab1:	subi	r0,r0,1
		st		(r1),r0

hlab2:	ld      r1,(r7+1)     		; R1, R0 <- stack
        ld      r0,(r7)
        addi    r7,r7,2
		reti     
		
;--------------------------------------
;-----------   MAIN   -----------------
;-------------------------------------- 

restart:

		ldpc	r7
		word	0xFFFD	; Pila al final de la RAM, espacio para 3 variables

		ldi		r1,SPICTRL
		ldi		r0,0		; 8 bits, no SSx
		st		(r1),r0		
		ldpc	r0			; 8 bits, SS1 ON, max speed (12.5Mhz)
		word	0x100		
		st		(r1),r0

		ldi		r0,3		; CMD read
		adpc	r6,1		; Llamada a subrutina
		jr		SPIxfer	
		ldi		r0,0x09		; MSB
		adpc	r6,1
		jr		SPIxfer	
		ldi		r0,0xFA		; 
		adpc	r6,1
		jr		SPIxfer	
		ldi		r0,0x60		; LSB
		adpc	r6,1
		jr		SPIxfer	

		ldi		r1,cnth		; Nº de bytes de audio
		ldi		r0,88		; (len/64K)
		st		(r1),r0
		ldi		r1,cntl
		ldpc	r0
		word	10988		; (len%64K)
		st		(r1),r0

		ldi		r1,IRQEN	; Habilita IRQ HBLK
		ldi		r0,EHBLK
		st		(r1),r0

		ldi		r0,0	; Retardo para UART
		ldi		r1,16
del:	subi	r0,r0,1
		jnz		del
		subi	r1,r1,1
		jnz		del

		ldpc	r0		; Imprime cadena de caracteres TXT
		word	txtle	; empaquetada tipo Little-Endian
		adpc 	r6,1	; Llamada a subrutina
		jr		putsle		

;-------- Borra memoria
		ldi		r0,0
		ldpc	r1
		word	0x1000
	
cls1:	st		(r1),r0
		addi	r1,r1,1
		jnz		cls1

		ldpc	r1
		word	0x1000
		st		(r7+2),r1
		
		ldpc	r5			; y0 = 1
		word	Y0VAL
		ldpc	r0
		word	400
		st		(r7),r0		; iteraciones y

buc1:	ldpc	r4			; x0 = -2.34
		word	X0VAL	
		ldpc	r3			; iteraciones x		
		word	512
buc2:
		or		r0,r4,r4
		or		r1,r5,r5
		adpc	r6,1
		jr		mandel_iter
		
		ror		r0,r0
		ror		r0,r0
		ror		r0,r0
		ror		r0,r0
		ror		r0,r0
		
		ld		r1,(r7+1)	; pix4
		add		r0,r0,r0
		adc		r1,r1,r1
		add		r0,r0,r0
		adc		r1,r1,r1
		add		r0,r0,r0
		adc		r1,r1,r1
		add		r0,r0,r0
		adc		r1,r1,r1
		st		(r7+1),r1

		ldi		r0,DELTA		; dx
		add		r4,r4,r0
		subi	r3,r3,1

		tsti	r3,3			; store 4 pixels
		jnz		sig1
		ld		r0,(r7+2)
		st		(r0),r1
		addi	r0,r0,1
		st		(r7+2),r0
		
sig1:	or		r3,r3,r3
		jnz		buc2

		ldi		r0,DELTA		; dy
		sub		r5,r5,r0
		ld		r0,(r7)
		subi	r0,r0,1
		st		(r7),r0
		jnz		buc1

cuel:	jr		cuel


;-------------------------------------------------------------------
;		Multiplicación fraccional
; Datos de 16 bits con 11 bits fraccionales val = dato/2048
;		R0 = (R0 * R1) >> 11
;  Nota: el valor de R2 se conserva
;-------------------------------------------------------------------

fmul11:	subi	r7,r7,1
		st		(r7),r6
		or		r6,r0,r0
		jpl		fml2			; Multiplicando (R6) siempre positivo
		neg		r6,r6
		neg		r1,r1
fml2:	ldi		r0,0
		add		r6,r6,r6		; Bit 14 (*8)
		jpl		fml3
		add		r0,r1,r1
		add		r0,r0,r0
		add		r0,r0,r0
fml3:	add		r6,r6,r6		; Bit 13 (*4)
		jpl		fml5
		add		r0,r0,r1
		add		r0,r0,r1
		add		r0,r0,r1
		add		r0,r0,r1
fml5:	add		r6,r6,r6		; Bit 12 (*2)
		jpl		fml7
		add		r0,r0,r1
		add		r0,r0,r1
		jr		fml7

fml4:	shra	r1,r1			; Resto de bits
fml7:	add		r6,r6,r6
		jz		fml6
		jpl		fml4
		add		r0,r0,r1
		jr		fml4
fml6:	ld		r6,(r7)
		addi	r7,r7,1
		jind	r6


;-------------------------------------------------------------------
; Iteración Mandelbrot
;  Calcula Zr = Zr*Zr - Zi*Zi + Cr
;          Zi = 2*Zr*Zi + Ci
;    mientras (Zr*Zr + Zi*Zi)<4
;  Parámetros: R0: Cr, R1: Ci
;  Retorna el nº de iteraciones (max 255)
;-------------------------------------------------------------------

mandel_iter:
		subi	r7,r7,8
		st		(r7+4),r6
		st		(r7+5),r5
		st		(r7+6),r4
		st		(r7+7),r3
		; Variables i, zr2, zi2, zri en pila

		or		r4,r0,r0	; R4 = Cr
		or		r5,r1,r1	; R5 = Ci

		or		r2,r4,r4	; R2 = Zr = Cr
		or		r3,r5,r5	; R3 = Zi = Ci
		ldi		r0,0		; i = 0
		st		(r7),r0

mitr1:	or		r0,r2,r2	; zri=fmul11(zr,zi);
		or		r1,r3,r3
		adpc	r6,1
		jr		fmul11
		add		r0,r0,r0	; zri= 2*zr*zi
		st		(r7+3),r0

		or		r0,r2,r2	; zr2=fmul11(zr,zr);
		or		r1,r2,r2
		adpc	r6,1
		jr		fmul11
		st		(r7+1),r0

		or		r0,r3,r3	; zi2=fmul11(zi,zi);
		or		r1,r3,r3
		adpc	r6,1
		jr		fmul11
		st		(r7+2),r0

		ld		r1,(r7+1)	; escape: (zr2 + zi2) > (4<<11)
		add		r1,r1,r0
		ldpc	r6
		word	4<<11
		cmp		r1,r6
		jc		mitr9

		ld		r1,(r7+1)
		sub		r0,r1,r0
		add		r2,r0,r4	; Zr = Zr2 - Zi2 + Cr

		ld		r0,(r7+3)
		add		r3,r0,r5	; Zi = 2Zri + Ci

		ld		r0,(r7)		; i++
		addi	r0,r0,1
		st		(r7),r0
		ldi		r1,255
		cmp		r0,r1		; while (i<255)
		jnz		mitr1

mitr9:	ld		r0,(r7)		; return i
		ld		r3,(r7+7)
		ld		r4,(r7+6)
		ld		r5,(r7+5)
		ld		r6,(r7+4)
		addi	r7,r7,8
		jind	r6



;----------------------------------------------------
; Envía cadena de caracteres empaquetados al terminal
; parámetros:
;	R0: puntero a la cadena de caracteres
;	R6: dirección de retorno
; retorna:
; 	R0-R2: modificados

putsle:	subi	r7,r7,1
		st		(r7),r6

		or		r2,r0,r0
putsle1:
		ld		r0,(r2)
		ldi		r1,0xff
		and		r0,r0,r1
		jz		putsle3
		adpc	r6,1
		jr		putch

		ld		r0,(r2)
		shr		r0,r0
		shr		r0,r0
		shr		r0,r0
		shr		r0,r0
		shr		r0,r0
		shr		r0,r0
		shr		r0,r0
		shr		r0,r0
		jz		putsle3
		adpc	r6,1
		jr		putch
		addi	r2,r2,1
		jr		putsle1

putsle3:
		ld		r6,(r7)
		addi	r7,r7,1
		jind	r6
;----------------------------------------------
; Envía caracter al terminal
; parámetros:
;	R0: dato a enviar
;	R6: dirección de retorno
; retorna:
; 	R1: modificado

putch:	ldi		r1,IOBASE	; Esperamos flag TBE en 1
		ld		r1,(r1+OPFLAGS)		
		tsti	r1,UARTTHRE
		jz		putch
		ldi		r1,IOBASE		
		st		(r1+OUARTDAT),r0		; envía dato
		jind	r6			; y retornamos


;----------------------------------------------
; Recibe caracter del terminal
; parámetros:
;	R6: dirección de retorno
; retorna:
;	R0: dato recibido
; 	R1: modificado

getch:	ldi		r1,IOBASE		;PFLAGS
getch1:	ld		r0,(r1+OPFLAGS)		; esperamos flag DV en 1
		tsti	r0,UARTDV
		jz		getch1
		ld		r0,(r1+OUARTDAT)		; leemos dato
		jind	r6			; y retornamos

;-----------------------------------------------
;       SPI
;-----------------------------------------------
;	Transferencia SPI
;    dato a transmitir: R0
;    dato recibido: R0
;    modifica R1
SPIxfer:
		ldi		r1,IOBASE
		st		(r1+OSPIDAT),r0
bspx1:	ld		r0,(r1+OPFLAGS)
		tsti	r0,SPIBUSY
		jnz		bspx1
		ld		r0,(r1+OSPIDAT)
		jind	r6			; retornamos	

txtle:	asczle	"\n\nGUS16 + VGA Mandelbrot\nJ. Arias 2022\n\n"

;-------- Final del código
endcode:

