//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
// CPU
//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
module CPU_1CV2(
output [15:0]a,		// Bus de direcciones
output [15:0]Do,	// Bus de datos de salida
//output rw,			// Escritura si 0, Lectura si 1
//output oe,			// Lectura para memorias asíncronas
output we,			// Escritura para memorias asíncronas
input  [15:0]di,	// Bus de datos de entrada 
input  clkin,		// Entrada de Reloj
input  waitin,		// Entrada de espera
input  resetb,		// Entrada de Reset, activa en bajo
input  irq,			// Entrada de interrupción
input  [2:0]ivector	// Vector de interrupción
);

wire [15:0]rega;
wire [15:0]regb;
wire [15:0]busd;
wire [2:0]aa;
wire [2:0]ba;
wire [2:0]da;
wire dwr;

// Lógica para parar el reloj
wire clk;
reg  waitr=1;
always @(negedge clkin) waitr <= ~waitin;
assign clk = waitr & clkin;

//assign we = st & (~clk);
//assign oe = (~st) ; //& (~clk);

assign we = st;

// Módulos de la CPU:

//------------ REGISTROS ---------------
REGBANK rbnk ( .a(rega),.b(regb),.d(busd),.asel(aa),.bsel(ba),.dsel(da),.wren(dwr),.clk(clk));
assign Do = regb;

// ---------- PC ----------
wire [15:0]regpc;
wire pcinc,ld,st,jmp,imm,irqstart;
wire [15:0]vector;
assign vector = {11'h000,ivector,2'b00};
assign pcinc = ~(ld|st|(irqstart&(~ldpc))); // Incrementa PC

REGPC pc ( .q(regpc), .d(busd), .resb(resetb), .vector(vector), .load(jmp), .inc(pcinc),
		   .clk(clk), .mode(mode), .reti(reti) );

// ---------- ALU ------------
wire [15:0]alua;
wire [15:0]alub;
wire cd,vd,zd,nd;
wire [1:0]aluop;
wire zab,ib,xa,ror,pca;
wire mode;

assign a = (ld | st) ? rega: regpc;

assign alua = pca ? regpc : rega;
assign alub = (ld | ldpc) ? di : (imm ? busimm : regb);

ALU alu ( .y(busd),.co(cd),.v(vd),.z(zd),.n(nd),.a(alua),.b(alub),
		   .op(aluop),.cf(cf),.zab(zab),.ib(ib),.ror(ror),.xa(xa) );

wire cf,vf,zf,nf;
wire wrcv,wrzn;
FLAGS flagr ( .qc(cf),.qv(vf),.qz(zf),.qn(nf), .dc(cd),.dv(vd),.dz(zd),.dn(nd),
              .wrzn(wrzn), .wrcv(wrcv), .clk(clk), .mode(mode) );

wire [15:0]busop;
wire opval,opvali;
assign opvali=~(ld | ldpc | st | jmp | irqstart);	// Entrada de op-code válido

IR ir( .q(busop), .opval(opval), .d(di), .opvali(opvali), .clk(clk), .resb(resetb) );

wire [15:0]busimm;
wire ldi, ldpc, reti;
IMM immdat( .f(busimm), .op(busop), .ldi(ldi) );

DECODING decoder(.aa(aa),.ba(ba),.da(da),.dwr(dwr),
				 .jmp(jmp),.ld(ld),.st(st),.ldpc(ldpc),.ldi(ldi),.pca(pca),
				 .imm(imm),.wrcv(wrcv),.wrzn(wrzn),
                 .fs(aluop),.zab(zab),.ib(ib),.xa(xa),.ror(ror), .reti(reti),
				 .op(busop),.cf(cf),.vf(vf),.zf(zf),.nf(nf), .opval(opval) );

assign rw = ~st;

// Interrupciones
IRQLOGIC irqlogic(.irq(irq), .reti(reti), .irqstart(irqstart), .mode(mode),
				  .resb(resetb), .clk(clk)  );

endmodule
//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------


//------------------------------------------------------------------------------------------
// ALU
//------------------------------------------------------------------------------------------

module ALU (
output [15:0]y,		// Salida de resultado
output co,			// Salida de acarreo
output v,			// Salida de overflow
output z,			// Salida de cero
output n,			// Salida de negativo
input [15:0]a,		// Entrada de operando A
input [15:0]b, 		// Entrada de operando B
input [1:0]op,		// Operación (0: suma, 1: XOR, 2: OR, 3: AND)
input cf, 			// Entrada de acarreo
input zab,			// Forzar operando A=0 si 0
input ib,			// Invertir bits de operando B si 1
input ror,			// Rotar a derecha si 1
input xa			// Operaciones con acarreo extendio (ADC,...)
);

wire c0;		// acarreo de entrada
assign c0 = xa ? cf : ib;

// Datos de entrada a sumador
wire [15:0]sa;		
wire [15:0]sb;
assign sa= zab ? a  : 16'b0;
assign sb= ib  ? ~b : b;

// Operación ALU interna
reg [15:0]f;	// No realmente registros
reg c15;
always@*
  case (op)
     0 : {c15,f} = sa+sb+c0;			// SUMA
     1 : {c15,f} = {1'bx,sa ^ sb};		// XOR
     2 : {c15,f} = {1'bx,sa | sb};		// OR
     3 : {c15,f} = {1'bx,sa & sb};		// AND
  endcase
// Flag de overflow
//assign v = (~(sb[15]^sa[15]))&(sa[15]^f[15]);
assign v = ((~sb[15])&(~sa[15])&f[15]) | (sb[15]&sa[15]&(~f[15]));

// Rotaciones
wire rb15;	// Bit para desplazar a la posición MSB del resultado
assign rb15 = xa ? cf : b[15]&ib;

assign y = ror ? {rb15,b[15:1]} : f;
assign co = ror ? b[0] : c15;

// Flags Z, N
assign z = (y==0);
assign n = y[15];

endmodule


//------------------------------------------------------------------------------------------
// Banco de Registros
//------------------------------------------------------------------------------------------

module REGBANK (
output [15:0]a,		// Salida para bus A (ALU, dir memoria)
output [15:0]b,		// Salida para bus B (ALU, datos memoria)
input [15:0]d, 		// Entrada de datos al banco
input [2:0]asel,	// Selección de registro para lectura a bus A
input [2:0]bsel,	// Selección de registro para lectura a bus B
input [2:0]dsel,	// Selección de registro para escritura
input wren,			// Habilitación de escritura si 1
input clk
);

wire [7:0]wr;
assign wr[0]=wren & (~dsel[2]) & (~dsel[1]) & (~dsel[0]);
assign wr[1]=wren & (~dsel[2]) & (~dsel[1]) & ( dsel[0]);
assign wr[2]=wren & (~dsel[2]) & ( dsel[1]) & (~dsel[0]);
assign wr[3]=wren & (~dsel[2]) & ( dsel[1]) & ( dsel[0]);
assign wr[4]=wren & ( dsel[2]) & (~dsel[1]) & (~dsel[0]);
assign wr[5]=wren & ( dsel[2]) & (~dsel[1]) & ( dsel[0]);
assign wr[6]=wren & ( dsel[2]) & ( dsel[1]) & (~dsel[0]);
assign wr[7]=wren & ( dsel[2]) & ( dsel[1]) & ( dsel[0]);

// Registros R0-R7
wire [15:0]q0;
wire [15:0]q1;
wire [15:0]q2;
wire [15:0]q3;
wire [15:0]q4;
wire [15:0]q5;
wire [15:0]q6;
wire [15:0]q7;

REG16 r0(.q(q0),.d(d),.wr(wr[0]),.clk(clk));
REG16 r1(.q(q1),.d(d),.wr(wr[1]),.clk(clk));
REG16 r2(.q(q2),.d(d),.wr(wr[2]),.clk(clk));
REG16 r3(.q(q3),.d(d),.wr(wr[3]),.clk(clk));
REG16 r4(.q(q4),.d(d),.wr(wr[4]),.clk(clk));
REG16 r5(.q(q5),.d(d),.wr(wr[5]),.clk(clk));
REG16 r6(.q(q6),.d(d),.wr(wr[6]),.clk(clk));
REG16 r7(.q(q7),.d(d),.wr(wr[7]),.clk(clk));

// Multiplexores para buses A y B
reg [15:0]a;	// No son registros
reg [15:0]b;
always@*
  case (asel)
     0 : a <= q0;
     1 : a <= q1;
     2 : a <= q2;
     3 : a <= q3;
     4 : a <= q4;
     5 : a <= q5;
     6 : a <= q6;
     7 : a <= q7;
  endcase
always@*
  case (bsel)
     0 : b <= q0;
     1 : b <= q1;
     2 : b <= q2;
     3 : b <= q3;
     4 : b <= q4;
     5 : b <= q5;
     6 : b <= q6;
     7 : b <= q7;
  endcase

endmodule

// Registro simple con habilitación de escritura para instanciar en el banco
module REG16 (output [15:0]q, input [15:0]d, input wr, input clk);
reg [15:0]q=0;
always @(posedge clk) q<= wr ? d : q;
endmodule

// Registro Contador de programa
module REGPC (output [15:0]q, input [15:0]d, input [15:0]vector, input resb,
			  input load, input inc, input mode, input clk, input reti);
reg [15:0]q0=0;	// PC modo normal
reg [15:0]q1;	// PC modo IRQ
wire [15:0]li;		// entrada a PC (normal o IRQ)
wire [15:0]inco;	// PC (normal o IRQ) incrementado

assign inco = q + inc;			// incrementador
assign li = load ? d : inco;	// mux load / inc

// registro PC modo normal
always @(posedge clk or negedge resb ) 
	if (!resb) q0<=16'h0000;
	else q0<= mode ? q0 : li;

// registro PC modo IRQ
always @(posedge clk ) 
	q1<= (mode&(~reti)) ? li : vector;

assign q = mode ? q1 : q0; 	// mux de salida

endmodule

//------------------------------------------------------------------------------------------
// Operandos inmediatos
//------------------------------------------------------------------------------------------

module IMM(
output [15:0]f,		// Salida de operando inmediato
input [15:0]op,		// Entrada desde reg. de instrucción
input ldi			// instrucción LDI
);

// Salida de operando inmediato
assign f = op[15] ? {op[11],op[11],op[11],op[11],op[11:0]} :
           ( ldi ? {8'h00,op[7:0]}: {12'h000,op[3:0]});

endmodule

//------------------------------------------------------------------------------------------
// Registro de instrucción
//------------------------------------------------------------------------------------------

module IR (
output [15:0]q,		// Salida (hacia operandos immediatos y decodificación)
output opval,		// Salida de op-code válido
input [15:0]d,		// Entrada (desde bus de datos)
input opvali,		// Entrada de op-code válido
input clk,
input resb
);
reg [15:0]q;
reg opval=0;
always @(posedge clk) q<= d;
always @(posedge clk or negedge resb) 
	if (!resb) 	opval<=1'b0;
	else 		opval<= opvali;
endmodule

//------------------------------------------------------------------------------------------
// FLAGS
//------------------------------------------------------------------------------------------
module FLAGS (
output qc,		// Salida de flag C
output qv,		// Salida de flag V
output qz,		// Salida de flag Z
output qn,		// Salida de flag N
input dc,		// Entrada de flag C
input dv,		// Entrada de flag V
input dz,		// Entrada de flag Z
input dn,		// Entrada de flag N
input wrzn,		// Escribir flags Z y N si 1
input wrcv,		// Escribir flags C y V si 1
input mode,		// modo normal o IRQ
input clk
);

reg qc0=0,qv0=0,qz0=0,qn0=0;
reg qc1=0,qv1=0,qz1=0,qn1=0;
wire wc0,wc1,wz0,wz1;

assign wc0 = wrcv & (~mode);
assign wz0 = wrzn & (~mode);
assign wc1 = wrcv & ( mode);
assign wz1 = wrzn & ( mode);

always @(posedge clk) begin
	qc0 <= wc0 ? dc: qc0;
	qv0 <= wc0 ? dv: qv0;
	qz0 <= wz0 ? dz: qz0;
	qn0 <= wz0 ? dn: qn0;

	qc1 <= wc1 ? dc: qc1;
	qv1 <= wc1 ? dv: qv1;
	qz1 <= wz1 ? dz: qz1;
	qn1 <= wz1 ? dn: qn1;
end

assign qc = mode ? qc1 : qc0;
assign qv = mode ? qv1 : qv0;
assign qz = mode ? qz1 : qz0;
assign qn = mode ? qn1 : qn0;

endmodule

//------------------------------------------------------------------------------------------
// Decodificación de instrucciones
//------------------------------------------------------------------------------------------
module DECODING(
output [2:0]aa,		// Selección de registro para lectura a bus A
output [2:0]ba,		// Selección de registro para lectura a bus B
output [2:0]da,		// Selección de registro para escritura
output dwr,			// Habilitación de escritura en registro
output jmp,			// Instrucción de salto (escribir PC)
output ld,			// Instrucción LD (lectura de memoria)
output st,			// Instrucción ST (escritura en memoria)
output ldi,			// Instrucción LDI 
output ldpc,		// Instrucción LDPC
output imm,			// Operandos inmediatos a bus B si 1
output pca,			// PC -> ALU_A si 1
output wrcv,		// Escribir en flags C y V
output wrzn,		// Escribir en flags Z y N
output [1:0]fs,		// Operación de la ALU
output zab,			// Forzar entrada A a 0 en la ALU si 0
output ib,			// Invertir bits de entrada B de la ALU si 1
output xa,			// Instrucciones extendidas (con acarreo)
output ror,			// Rotaciones/desplazamientos a derechas
output reti,		// Retorno de interrupción (se ejecuta como JIND)

input [15:0]op,		// Código de operación desde registro de instrucción
input cf,			// Entrada de Flag de Acarreo
input vf,			// Entrada de Flag de Overflow
input zf,			// Entrada de Flag de Zero
input nf,			// Entrada de Flag de Signo
input opval			// Entrada de instrucción válida

);

// multiplexor de 8 a 1 para saltos
wire jr;
wire [3:0]mxjr1;
wire [1:0]mxjr2;
assign mxjr1 = op[14] ? {1'b1,vf,~nf,nf} : {~cf,cf,~zf,zf};
assign mxjr2 = op[13] ? mxjr1[3:2] : mxjr1[1:0];
assign jr    = op[12] ? mxjr2[1]   : mxjr2[0];

// Selección de registros
assign aa = op[6:4];
assign ba = op[2:0];
assign da = op[10:8];

// Ciertos OP-codes y líneas de control decodificados directamente
wire jind;
assign jind = opval & (~op[15]) & op[14] & op[13] & (~op[12]) & op[11] & op[7];
assign jmp = (opval & op[15] & jr) | jind;
assign ld  = opval & (~op[15]) & op[14] & op[13] & (~op[12]) & (~op[11]) & (~op[7]);
assign st  = opval & (~op[15]) & op[14] & op[13] & (~op[12]) & (~op[11]) & op[7];
assign ldpc= opval & (~op[15]) & op[14] & op[13] &   op[12]  & (~op[11]);
assign ldi = opval & (~op[15]) & op[14] & op[13] &   op[12]  &   op[11];
assign reti= opval & (~op[15]) & op[14] & op[13] & (~op[12]) &   op[11]  & op[7] & op[3];

// Tabla lógica del resto de las salidas del decodificador
reg [1:0]fs;								// Función de la ALU, no es registro
reg zab, ib, xa, ror, imm, wd, wc, wz, pca; // salidas de control, nos son registros
always@*
 casex ({op[15:11],op[7]})
                                               //   fs  zab   ib   xa  ror   imm   wd   wc   wz  pca
 6'b000000:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b0,1'b0,1'b0, 1'b0,1'b1,1'b1,1'b1,1'b0};	// ADD
 6'b000001:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b0,1'b0,1'b0, 1'b1,1'b1,1'b1,1'b1,1'b0};	// ADDI
 6'b000010:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b0,1'b1,1'b0, 1'b0,1'b1,1'b1,1'b1,1'b0};	// ADC
 6'b000011:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b0,1'b1,1'b0, 1'b1,1'b1,1'b1,1'b1,1'b0};	// ADCI

 6'b000100:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b1,1'b0,1'b0, 1'b0,1'b1,1'b1,1'b1,1'b0};	// SUB
 6'b000101:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b1,1'b0,1'b0, 1'b1,1'b1,1'b1,1'b1,1'b0};	// SUBI
 6'b000110:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b1,1'b1,1'b0, 1'b0,1'b1,1'b1,1'b1,1'b0};	// SBC
 6'b000111:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b1,1'b1,1'b0, 1'b1,1'b1,1'b1,1'b1,1'b0};	// SBCI

 6'b001000:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b1,1'b0,1'b0, 1'b0,1'b0,1'b1,1'b1,1'b0};	// CMP
 6'b001001:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b1,1'b0,1'b0, 1'b1,1'b0,1'b1,1'b1,1'b0};	// CMPI
 6'b001010:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b11,1'b1,1'b0,1'bx,1'b0, 1'b0,1'b1,1'b0,1'b1,1'b0};	// AND
 6'b001011:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b11,1'b1,1'b0,1'bx,1'b0, 1'b1,1'b1,1'b0,1'b1,1'b0};	// ANDI

 6'b001100:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b11,1'b1,1'b0,1'bx,1'b0, 1'b0,1'b0,1'b0,1'b1,1'b0};	// TST
 6'b001101:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b11,1'b1,1'b0,1'bx,1'b0, 1'b1,1'b0,1'b0,1'b1,1'b0};	// TSTI
 6'b001110:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b10,1'b1,1'b0,1'bx,1'b0, 1'b0,1'b1,1'b0,1'b1,1'b0};	// OR
 6'b001111:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b10,1'b1,1'b0,1'bx,1'b0, 1'b1,1'b1,1'b0,1'b1,1'b0};	// ORI

 6'b010000:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b01,1'b1,1'b0,1'bx,1'b0, 1'b0,1'b1,1'b0,1'b1,1'b0};	// XOR
 6'b010001:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b01,1'b1,1'b0,1'bx,1'b0, 1'b1,1'b1,1'b0,1'b1,1'b0};	// XORI
 6'b010010:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b10,1'b0,1'b1,1'bx,1'b0, 1'b0,1'b1,1'b0,1'b1,1'bx};	// NOT
 6'b010011:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b0,1'b1,1'b0,1'b0, 1'b0,1'b1,1'b0,1'b1,1'bx};	// NEG

 6'b010100:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'bxx,1'bx,1'b0,1'b0,1'b1, 1'b0,1'b1,1'b1,1'b1,1'b0};	// SHR
 6'b010101:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'bxx,1'bx,1'b1,1'b0,1'b1, 1'b0,1'b1,1'b1,1'b1,1'b0};	// SHRA
 6'b010110:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'bxx,1'bx,1'b1,1'b1,1'b1, 1'b0,1'b1,1'b1,1'b1,1'b0};	// ROR

//6'b010111:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'bxx,1'bx,1'bx,1'bx,1'bx, 1'bx,1'bx,1'bx,1'bx,1'b0};	// ILEG

 6'b011000:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b10,1'b0,1'b0,1'bx,1'b0, 1'b0,1'b1,1'b0,1'b1,1'bx};	// LD
 6'b011001:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'bxx,1'bx,1'bx,1'bx,1'bx, 1'bx,1'b0,1'b0,1'b0,1'b0};	// ST
 6'b011010:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b0,1'b0,1'b0, 1'b1,1'b1,1'b0,1'b0,1'b1};	// ADPC
 6'b011011:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b10,1'b0,1'b0,1'bx,1'b0, 1'b0,1'b0,1'b0,1'b0,1'bx};	// JIND

 6'b01110?:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b10,1'b0,1'b0,1'bx,1'b0, 1'b0,1'b1,1'b0,1'b0,1'bx};	// LDPC
 6'b01111?:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b10,1'b0,1'b0,1'bx,1'b0, 1'b1,1'b1,1'b0,1'b0,1'bx};	// LDI

 6'b1?????:{fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'b00,1'b1,1'b0,1'b0,1'b0, 1'b1,1'b0,1'b0,1'b0,1'b1};	// JRx

  default: {fs,zab,ib,xa,ror,imm,wd,wc,wz,pca}<={2'bxx,1'bx,1'bx,1'bx,1'bx, 1'bx,1'bx,1'bx,1'bx,1'bx};
 endcase
// Escrituras en registros
assign dwr  = opval & wd;	// banco de registros
assign wrcv = opval & wc;	// Flags C y V
assign wrzn = opval & wz;	// Flags Z y N

endmodule

module IRQLOGIC (
	input 	irq,		// Petición de interrupción
	input	reti,		// final de interrupción
	output 	irqstart,	// pulso de comienzo de IRQ
    output 	mode,		// modo del micro: normal o IRQ
	input	resb,		// reset, activo en bajo
	input 	clk			
);

reg q0=0, mode=0;
wire ireti;
assign ireti=reti&(~irq);

always @(posedge clk or negedge resb ) 
	begin
	if (!resb) begin 
		q0<=1'b0;
		mode<=1'b0;
	end
	else begin
		q0 <= (~ireti) & (q0 | irq);
		mode <= (~ireti) & q0;
	end
	end

assign irqstart = (~mode) & q0;

endmodule

