//--------------------------------------------------------------------
// System CPC464
//--------------------------------------------------------------------

module SYSTEM (
	input clk,
	input reset,
	// External memory
	output xwe,			// Write Enable (active low)
	output xoe,			// Output Enable (active low)
	output [15:0]xa,	// address
	output [15:0]xdo,	// data output
	input  [15:0]xdi,	// data input
	output xbhe,		// Byte High Enable (active low)
	output xble,		// Byte Low Enable (active low)
	// UART (not implemented yet)
	output txd,
	input rxd,
	// VIDEO
	output hsyn,
	output vsyn,
	output [3:0]r,
	output [3:0]g,
	output [3:0]b,
	// INPUT / OUTPUT (SPI signals)
	input  [7:0]pinin,
	output [7:0]pinout,
	// AUDIO
	output	pwmout,	
	// JOYSTICK
`ifdef NES
	output	gclk,
	output	glat,
	input	gin,
`endif
	// PS2 KEYBOARD
	input	kclk,
	input	kdat,
	// for scope probes
	output [1:0]debug	
);

wire resetg;	// Global reset
assign resetg=reset | resetk;

//////////////////////////// Clocking //////////////////////////////
reg [4:0]clkdiv=0;	// Clock divider
reg ck4m=1;			// 4 pulses / us
wire ckfast;		// up to 25MHz (*7/8 during active video)
wire cke_psg;		// clock enable for PSG (1 MHz = 1 pulse / us)

assign ckfast=vrd|clk;

always @(posedge clk) begin
	if (clkdiv==24) clkdiv<=0;
	else clkdiv<=clkdiv+1;
	if ((clkdiv==0)|(clkdiv==6)|(clkdiv==12)|(clkdiv==18)) ck4m<=0;
	if ((~vrd)&(~ck4m)) ck4m<=1;
end
wire cclk;	// CPU clock
assign cclk=(fast|motor) ? ckfast : (~((~vrd)&(~ck4m)))|clk;
assign cke_psg=(clkdiv==0);

/////////////////////////////// CPU ////////////////////////////////
wire reset_n,wait_n,int_n,nmi_n,busrq_n,cen;
wire m1_n,mreq_n,iorq_n,rd_n,wr_n,rfsh_n,halt_n,busak_n;
wire [7:0]cdi;	// CPU Data bus, Input
wire [7:0]cdo;	// CPU Data bus, Output
wire [15:0]ca;	// CPU Address bus

assign reset_n=~resetg;
// always inactive signals
assign wait_n=1'b1;
assign nmi_n=1'b1;
assign busrq_n=1'b1;
// always active signals
assign cen=1'b1;		// Clock enable

tv80s Z80CPU(
  // Outputs
  m1_n, mreq_n, iorq_n, rd_n, wr_n, rfsh_n, halt_n, busak_n, ca, cdo,
  // Inputs
  reset_n, cclk, wait_n, int_n, nmi_n, busrq_n, cdi, cen
  );

//// Input bus MUX
reg [7:0]iodi;	// data from preripherals (not a register)
assign cdi = (iorq_n)?  (romcs ? dorom : mxdi) : (iodi);

//// Perif. MUX
always@*
 casex (ca[15:8])
 	8'b11101111: iodi<=boot ? {bootreg[7:1],pinin[7]} : 8'b1111111; // boot reg / printer
 	8'b111101xx: iodi<=ppido;
 	8'b10111111: iodi<=crtcdo;
 	default: iodi<=8'hFF;
 endcase

////////////////// External memory (as 128K x 8bits) ////////////////////

wire [16:0]mxa;	// Ext. mem. address (17 bits)
wire [7:0]mxdi;	// Input data (8 bits)

assign xbhe=~mxa[0];
assign xble=mxa[0];
assign xoe= ~(vrd|(~rd_n));
assign xwe= ~((~vrd)&(~wr_n)&(~mreq_n))|cclk;
assign xa=mxa[16:1];
assign xdo={cdo,cdo};
assign mxdi=mxa[0]?xdi[15:8]:xdi[7:0];

// ROM banking emulation via A16
wire ma16;	// CPC ROMS (2x16K) must be preloaded at 128K and (128K+48K)
assign ma16= ((~rd_n)&(~ca[15])&(~ca[14])&lorome) |
			 ((~rd_n)&( ca[15])&( ca[14])&hirome) | a16;
			 
assign mxa=vrd ? {1'b0,va} : {ma16,ca};

//////////////// BOOT ROM //////////////////
wire romcs;
wire [7:0]dorom;
assign romcs=(~mreq_n)&(~rd_n)&(~ca[15])&(~ca[14])&(~ca[13])&boot;

genrom #(
	.INITFILE("romBOOT.hex"),
	.AW(9),
	.DW(8)
	) rom0 (.clk(~cclk),.addr(ca[8:0]),.data_out(dorom));


///////////// Boot register (same I/O than printer during boot) /////////////

reg [7:0]bootreg=8'hFE;
wire mosi,sck,cs1,cs2,miso;
wire boot,a16,fast;
assign mosi=bootreg[7];
assign sck =bootreg[6];
assign cs1 =bootreg[5];
assign cs2 =bootreg[4];
assign boot=bootreg[3];
assign a16 =bootreg[2];
assign fast=bootreg[1];

always @(posedge cclk or posedge resetg) 
  if (resetg) bootreg<=8'hFE; 
  else if ((~iorq_n)&(~wr_n)&(~ca[12])&boot) bootreg<=cdo[7:0];
// SPI
//assign pinout[7:4]=bootreg[7:4];
assign miso=pinin[7];

////////////////////////////////////////////////////////////////////////
////////////////////////////// Video ///////////////////////////////////

wire vrd;			// Video Read external memory
wire [5:0]vout;		// Video output (2 bits, thermometer code, R, G, B)
wire [3:0]palix; 	// Palette index
wire [4:0]paldat;	// Palette data
wire palwr;			// Palete write
wire [15:0]va;		// Video Address
wire avsyn;			// Amstrad Vsyncs
wire dvsyn;

videoCPC video0 (.clk(clk), .di(mxdi), .hmax(crtc1), .vmax(crtc6),
			  .ma({crtc12,crtc13}), .va(va), .vrd(vrd),
			  .hsyn(hsyn),.vsyn(vsyn),.video(vout),.mode(videomode),
			  .palix(palix), .paldat(paldat), .palwr(palwr),
			  .border(border),
			  .avsyn(avsyn), .dvsyn(dvsyn)
			   );
// 6-bit to 12-bit mapping
wire [3:0]cpcR;
wire [3:0]cpcG;
wire [3:0]cpcB;
assign cpcB={vout[1],vout[0],vout[0],vout[0]};
assign cpcG={vout[3],vout[2],vout[2],vout[2]};
assign cpcR={vout[5],vout[4],vout[4],vout[4]};

// Gate Array registers (Write only)
reg [4:0]pen=0;
reg [4:0]border=0;
reg [3:0]gacontrol=0;
wire lorome,hirome;
wire [1:0]videomode;
assign lorome=~gacontrol[2];
assign hirome=~gacontrol[3];
assign videomode=gacontrol[1:0];

assign palix=pen[3:0];
assign paldat=cdo[4:0];
assign palwr=(~iorq_n)&(~wr_n)&(~ca[15])&(~cdo[7])&( cdo[6])&(~pen[4]);

always @(posedge cclk ) begin
	if ((~iorq_n)&(~wr_n)&(~ca[15])&(~cdo[7])&(~cdo[6])) pen<=cdo[4:0];
	if ((~iorq_n)&(~wr_n)&(~ca[15])&(~cdo[7])&( cdo[6]) & pen[4]) border<=cdo[4:0];
	if ((~iorq_n)&(~wr_n)&(~ca[15])&( cdo[7])&(~cdo[6])) gacontrol<=cdo[3:0];
end

// Gate Array Interrupt
reg [6:0]icnt=0;// Counter (1 bit more than GA due to double number of lines)
reg ohsyn;		// For edge detection
reg ovsyn;
reg irq=0;		// Interrupt Request Flip flop
wire intack;	// CPU Interrupt Acknowledge
wire resint;	// Reset Interrupt Counter from Gate Array

assign intack=~(iorq_n | m1_n);
assign resint=((~iorq_n)&(~wr_n)&(~ca[15])&( cdo[7])&(~cdo[6]))&(cdo[4]);

always @(posedge cclk) begin
	ohsyn<=hsyn;
	ovsyn<=dvsyn;
	if (resint) icnt<=0;
	else if (intack) begin irq<=0; icnt[6]<=1'b0; end
	if (dvsyn&(~ovsyn)) begin icnt<=7'd0; irq<=1; end
	else if (hsyn & (~ohsyn)) begin
			if (icnt==104) begin icnt<=0; irq<=1; end
			else icnt<=icnt+1; 
		end
end

assign int_n = ~irq;

// 6845 registers (Write)
reg [4:0]crtcix=0;		// Register Index
reg [5:0]crtc1=0;		// Reg #1: Horizontal Displayed (*16 mode 2 pixels)
reg [4:0]crtc6=0;		// Reg #6: Vertical Displayed (*16)
reg [7:0]crtc12=8'h00;	// Reg #12: Display Start Address (high)
reg [7:0]crtc13=8'h00;	// Reg #13: Display Start Address (low)
always @(posedge cclk) begin
	if ((~iorq_n)&(~wr_n)&(~ca[14])&(~ca[9])&(~ca[8])) crtcix<=cdo[4:0];
	if ((~iorq_n)&(~wr_n)&(~ca[14])&(~ca[9])&( ca[8])) begin
		if (crtcix==5'd1)  crtc1<=cdo[5:0];
		if (crtcix==5'd6)  crtc6<=cdo[4:0];
		if (crtcix==5'd12) crtc12<=cdo;
		if (crtcix==5'd13) crtc13<=cdo;
	end
end

wire [7:0]crtcdo;	// register read
assign crtcdo=(crtcix[0])? crtc13 : crtc12;

//////////////////////////// 8255 PPI /////////////////////////////
// Outputs
reg [7:0]ppia;	// goes to PSG data bus
reg [7:0]ppic;	// Keyboard row select, PSG control, cassette control
wire motor,tapeout,tapein;
assign motor=ppic[4];
assign tapeout=ppic[5];

always @(posedge cclk) begin
	if ((~iorq_n)&(~wr_n)&(~ca[11])&(~ca[9])&(~ca[8])) ppia<=cdo;
	if ((~iorq_n)&(~wr_n)&(~ca[11])&( ca[9])&(~ca[8])) ppic<=cdo;
	if ((~iorq_n)&(~wr_n)&(~ca[11])&( ca[9])&( ca[8])) begin
		if (~cdo[7]) ppic[cdo[3:1]]<=cdo[0];	// single pin write
	end
end
// Inputs
reg [7:0]ppido;	// PPI output data bus
wire [7:0]ppib;	// PPI port B: VSYN, LK jumpers, Printer Busy, Cassete input

assign ppib={tapein,6'b000111,avsyn};

always @*
  case (ca[9:8])
  2'b00: ppido<=psgdo;	// Data read from PSG
  2'b01: ppido<=ppib;
  2'b10: ppido<=ppic;
  2'b11: ppido<=8'h00;	// PPI control is write only
  endcase

//////////////////// AY-3-8912 PSG //////////////////////
// actually a YM2149 recreation by José Tejada (JT49)
// modiffied for the reduction of tables and output range

wire [7:0]psgdo;	// data out bus to ppiA
wire [7:0]audio;	// 8-bit audio

jt49_bus ay_3_8912 (
	.rst_n(~resetg), .clk(clk), .clk_en(cke_psg),
	.bdir(ppic[7]), .bc1(ppic[6]), .din(ppia),
	.sel(1'b1), .dout(psgdo), .sound(audio),
	.IOA_in(keydat)
);

// 8-bit PWM output
reg [7:0]pwcnt=0;	// PWM counter
reg [7:0]pbuf=0;	// buffered audio, 8 bits

reg pwmout=0;		// Audio output

always @(posedge clk) begin
	pwmout<=(pwcnt==pbuf)? 0 : ((pwcnt==0)? 1 : pwmout);
	if (pwcnt==254) begin 
		pbuf<=audio;
		pwcnt<=0;
	end else pwcnt<=pwcnt+1;
end

//////////////////// KEYBOARD //////////////////////
wire [7:0]keydat;
wire resetk;		// keyboard generated reset

keybps2 keyboard( .clk(cclk), .kclk(kclk|grab), .kdat(kdat|grab),
 				  .a(ppic[3:0]), .row(keydat), .resetk(resetk) );

//////////////////// Tape Player Computer //////////////////////
// Video sharing
wire video,rem, viden;
assign r=viden? {video,video,video,video} :cpcR;
assign g=viden? {(rem)^video,video,video,video} :cpcG;
assign b=viden? ({1'b0,(~video)^rem,1'b0,(~video)^rem}) :cpcB;

// SPI sharing
wire tsck,tmosi,tcs1b,tcs2b;
assign pinout[7:4]=boot? bootreg[7:4] : {tmosi,tsck,tcs1b,tcs2b};

// Keyboard
wire grab;
// Sampling clock (derived from Z80 clock, 44.1KHz)
reg [6:0]ck44div=0;
always @(posedge cclk) ck44div<=(ck44div==7'd90)? 0 : ck44div+1;

Tapeplayer tapplay( .clk(clk), .reset(boot), .hsyn(hsyn), .vsyn(vsyn),
		.video(video), .rem(rem), .viden(viden),
		.miso(miso), .sck(tsck), .mosi(tmosi), .cs1b(tcs1b), .cs2b(tcs2b),
		.kclk(kclk), .kdat(kdat), .grab(grab),
		.ck44k(ck44div[6]), .tapeout(tapein), .tapein(tapeout), .motor(motor)
);


endmodule



//----------------------------------------------------------------------------
//-- Generic synchronous ROM memory
//----------------------------------------------------------------------------

module genrom (        
    input clk,	// data out on rising edges                  
    input wire [AW-1: 0] addr,      
    output reg [DW-1: 0] data_out   
);

parameter INITFILE = "rand.hex";
parameter AW = 13;
parameter DW = 8;

reg [DW-1: 0] rom [0: (1<<AW)-1];

// synchronous read
always @(posedge clk) begin
	data_out <= rom[addr];
end

//-- Load contents with hex file
initial begin
  $readmemh(INITFILE, rom);
end

endmodule

