	output "romTAP.bin"

;------- IO ports -------

CONTROL	equ	0
STATUS	equ	0
SPI	equ	1
KEYB	equ	2
RLE	equ	3

;------ Control bits ------
GRAB	equ	$80
SPIFAST	equ	$40
CS1B	equ	$20
CS2B	equ	$10
;------ Status bits -------
MOTOR	equ	$4
SPIBUSY	equ	$2
KEYVAL	equ	$1

;--------------------------------------
;--------------------------------------
;  CODE
;--------------------------------------
;--------------------------------------
	org	0

	di
	im	1
	jp	start

	block	8-$
spibyte:	; Also RST 8
	out	(SPI),a
spb1:	in	a,(STATUS)
	and	SPIBUSY
	jr	nz,spb1
	in	a,(SPI)
	ret

;--------------------------------------
; SD card selection
;

lcshl:	out	(CONTROL),a
	pop	af
	ret

	block	$18-$
SD_CS_L:	; Also RST $18
	push	af
	in 	a,(CONTROL)
	and	~CS2B
	jr	lcshl
	block	$20-$
SD_CS_H:	; Also RST $20
	push	af
	in 	a,(CONTROL)
	or	CS2B
	jr	lcshl

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	block	$38-$
INTerrupt:
	ex	af,af'
	exx
	ld	hl,(pbuf)
	ld	a,(rlewr)	; ==0 :read !=0 :write
	or	a
	jr	nz,INTwr
	ld	a,(hl)		;------ READ buffer --------
	dec	a
	out	(RLE),a		; put RLE data
inte3:	in	A,(STATUS)
	and	MOTOR
	jr	z,inte9
	inc	hl		; update pointer
	ld	a,$1C
	cp	h
	jr	nz,inte2
	ld	h,$18
inte2:	ld	(pbuf),hl
inte9:	exx			; return
	ex	af,af'
	ei
	reti			
				;------ WRITE to buffer -------
INTwr:	in	a,(RLE)		; get RLE data
	ld	(hl),a
	ld	a,$7f		; max count (just in case)
	out	(RLE),a	
	jr	inte3
	
	block	256-$

 db $00,$00,$00,$00,$00,$00,$00,$00 ;  
 db $30,$78,$78,$30,$30,$00,$30,$00 ; !
 db $6c,$6c,$6c,$00,$00,$00,$00,$00 ; "
 db $6c,$6c,$fe,$6c,$fe,$6c,$6c,$00 ; #
 db $30,$7c,$c0,$78,$0c,$f8,$30,$00 ; $
 db $00,$c6,$cc,$18,$30,$66,$c6,$00 ; %
 db $38,$6c,$38,$76,$dc,$cc,$76,$00 ; &
 db $60,$60,$c0,$00,$00,$00,$00,$00 ; '
 db $18,$30,$60,$60,$60,$30,$18,$00 ; (
 db $60,$30,$18,$18,$18,$30,$60,$00 ; )
 db $00,$66,$3c,$ff,$3c,$66,$00,$00 ; *
 db $00,$30,$30,$fc,$30,$30,$00,$00 ; +
 db $00,$00,$00,$00,$00,$30,$30,$60 ; ,
 db $00,$00,$00,$fc,$00,$00,$00,$00 ; -
 db $00,$00,$00,$00,$00,$30,$30,$00 ; .
 db $06,$0c,$18,$30,$60,$c0,$80,$00 ; /
 db $7c,$c6,$ce,$de,$f6,$e6,$7c,$00 ; 0
 db $30,$70,$30,$30,$30,$30,$fc,$00 ; 1
 db $78,$cc,$0c,$38,$60,$cc,$fc,$00 ; 2
 db $78,$cc,$0c,$38,$0c,$cc,$78,$00 ; 3
 db $1c,$3c,$6c,$cc,$fe,$0c,$1e,$00 ; 4
 db $fc,$c0,$f8,$0c,$0c,$cc,$78,$00 ; 5
 db $38,$60,$c0,$f8,$cc,$cc,$78,$00 ; 6
 db $fc,$cc,$0c,$18,$30,$30,$30,$00 ; 7
 db $78,$cc,$cc,$78,$cc,$cc,$78,$00 ; 8
 db $78,$cc,$cc,$7c,$0c,$18,$70,$00 ; 9
 db $00,$30,$30,$00,$00,$30,$30,$00 ; :
 db $00,$30,$30,$00,$00,$30,$30,$60 ; ;
 db $18,$30,$60,$c0,$60,$30,$18,$00 ; <
 db $00,$00,$fc,$00,$00,$fc,$00,$00 ; =
 db $60,$30,$18,$0c,$18,$30,$60,$00 ; >
 db $78,$cc,$0c,$18,$30,$00,$30,$00 ; ?
 db $7c,$c6,$de,$de,$de,$c0,$78,$00 ; @
 db $30,$78,$cc,$cc,$fc,$cc,$cc,$00 ; A
 db $fc,$66,$66,$7c,$66,$66,$fc,$00 ; B
 db $3c,$66,$c0,$c0,$c0,$66,$3c,$00 ; C
 db $f8,$6c,$66,$66,$66,$6c,$f8,$00 ; D
 db $fe,$62,$68,$78,$68,$62,$fe,$00 ; E
 db $fe,$62,$68,$78,$68,$60,$f0,$00 ; F
 db $3c,$66,$c0,$c0,$ce,$66,$3e,$00 ; G
 db $cc,$cc,$cc,$fc,$cc,$cc,$cc,$00 ; H
 db $78,$30,$30,$30,$30,$30,$78,$00 ; I
 db $1e,$0c,$0c,$0c,$cc,$cc,$78,$00 ; J
 db $e6,$66,$6c,$78,$6c,$66,$e6,$00 ; K
 db $f0,$60,$60,$60,$62,$66,$fe,$00 ; L
 db $c6,$ee,$fe,$fe,$d6,$c6,$c6,$00 ; M
 db $c6,$e6,$f6,$de,$ce,$c6,$c6,$00 ; N
 db $38,$6c,$c6,$c6,$c6,$6c,$38,$00 ; O
 db $fc,$66,$66,$7c,$60,$60,$f0,$00 ; P
 db $78,$cc,$cc,$cc,$dc,$78,$1c,$00 ; Q
 db $fc,$66,$66,$7c,$6c,$66,$e6,$00 ; R
 db $78,$cc,$e0,$70,$1c,$cc,$78,$00 ; S
 db $fc,$b4,$30,$30,$30,$30,$78,$00 ; T
 db $cc,$cc,$cc,$cc,$cc,$cc,$fc,$00 ; U
 db $cc,$cc,$cc,$cc,$cc,$78,$30,$00 ; V
 db $c6,$c6,$c6,$d6,$fe,$ee,$c6,$00 ; W
 db $c6,$c6,$6c,$38,$38,$6c,$c6,$00 ; X
 db $cc,$cc,$cc,$78,$30,$30,$78,$00 ; Y
 db $fe,$c6,$8c,$18,$32,$66,$fe,$00 ; Z
 db $78,$60,$60,$60,$60,$60,$78,$00 ; [
 db $c0,$60,$30,$18,$0c,$06,$02,$00 ; \ 
 db $78,$18,$18,$18,$18,$18,$78,$00 ; ]
 db $10,$38,$6c,$c6,$00,$00,$00,$00 ; ^
 db $00,$00,$00,$00,$00,$00,$00,$ff ; _
 db $30,$30,$18,$00,$00,$00,$00,$00 ; `
 db $00,$00,$78,$0c,$7c,$cc,$76,$00 ; a
 db $e0,$60,$60,$7c,$66,$66,$dc,$00 ; b
 db $00,$00,$78,$cc,$c0,$cc,$78,$00 ; c
 db $1c,$0c,$0c,$7c,$cc,$cc,$76,$00 ; d
 db $00,$00,$78,$cc,$fc,$c0,$78,$00 ; e
 db $38,$6c,$60,$f0,$60,$60,$f0,$00 ; f
 db $00,$00,$76,$cc,$cc,$7c,$0c,$f8 ; g
 db $e0,$60,$6c,$76,$66,$66,$e6,$00 ; h
 db $30,$00,$70,$30,$30,$30,$78,$00 ; i
 db $0c,$00,$0c,$0c,$0c,$cc,$cc,$78 ; j
 db $e0,$60,$66,$6c,$78,$6c,$e6,$00 ; k
 db $70,$30,$30,$30,$30,$30,$78,$00 ; l
 db $00,$00,$cc,$fe,$fe,$d6,$c6,$00 ; m
 db $00,$00,$f8,$cc,$cc,$cc,$cc,$00 ; n
 db $00,$00,$78,$cc,$cc,$cc,$78,$00 ; o
 db $00,$00,$dc,$66,$66,$7c,$60,$f0 ; p
 db $00,$00,$76,$cc,$cc,$7c,$0c,$1e ; q
 db $00,$00,$dc,$76,$66,$60,$f0,$00 ; r
 db $00,$00,$7c,$c0,$78,$0c,$f8,$00 ; s
 db $10,$30,$7c,$30,$30,$34,$18,$00 ; t
 db $00,$00,$cc,$cc,$cc,$cc,$76,$00 ; u
 db $00,$00,$cc,$cc,$cc,$78,$30,$00 ; v
 db $00,$00,$c6,$d6,$fe,$fe,$6c,$00 ; w
 db $00,$00,$c6,$6c,$38,$6c,$c6,$00 ; x
 db $00,$00,$cc,$cc,$cc,$7c,$0c,$f8 ; y
 db $00,$00,$fc,$98,$30,$64,$fc,$00 ; z
 db $1c,$30,$30,$e0,$30,$30,$1c,$00 ; {
 db $18,$18,$18,$00,$18,$18,$18,$00 ; |
 db $e0,$30,$30,$1c,$30,$30,$e0,$00 ; }
 db $76,$dc,$00,$00,$00,$00,$00,$00 ; ~
 db $aa,$55,$aa,$55,$aa,$55,$aa,$55  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;	
start:	
	ld	sp,$2000

	ld	a,$7f
	out	(CONTROL),a
	xor	a
	call	cls
	call	waitaction
	jp	filesel
	;jp	pruebas
	
;--------------------------------------------------
;--------------------------------------------------
;	SD Card init
;--------------------------------------------------
;--------------------------------------------------
init:	
ini1:	call	sd_init
	or	a
	jr	nz,inie1
	call	FAT_init
	or	a
	jr	nz,inie2
	ret
inie1:	push	af
	ld	hl,txterrsd
inie3:	ld	de,video+7
	call	puts
	ex	de,hl
	pop	af
	add	48
	ld	(hl),a
	ld	a,127
	inc	hl
	ld	(hl),a
	ld	de,video+32+7
	ld	hl,txterrlin
	call	puts
	ld	a,1
	ret
inie2:	push	af
	ld	hl,txterrfat
	jr	inie3
txterrsd:	db 127,"Error:  SDinit=",0
txterrfat:	db 127,"Error: FATinit=",0
txterrlin:	db 127,127,127,127,127,127,127,127,127
		db 127,127,127,127,127,127,127,127,127,0
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;	
; Pops to foreground if 
;  <sysreq> is pressed or
;  Tape motor starts
waitaction:
	xor	a
	call	cls
wa1:	in	a,(STATUS) ; detect rising edge of MOTOR
	ld	b,a
	ld	a,(ostat)
	cpl	
	and	b
	and	MOTOR	
	ld	a,b
	ld	(ostat),a
	jr	nz,wa9
	in	a,(STATUS)
	and	KEYVAL	; Key received
	jr	z,wa1
	call	getkey
	cp	$7c	; sys_req?
	jr	nz,wa1

wa9:	in	a,(CONTROL)	; grab keyboard
	or	GRAB
	out	(CONTROL),a
	ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Fill screen with the character in Acc
cls:	push	bc
	push	de
	push	hl
	ld	hl,video
	ld	de,video+1
	ld	bc,767
	ld	(hl),a
	ldir
	pop	hl
	pop	de
	pop	bc
	ret
	
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; HL: pointer to string
; DE: pointer to video
; HL, DE returns incremented
puts:	push	af
pts1:	ld	a,(hl)
	or	a
	jr	z,pts2
	inc	hl
	ex	de,hl
	ld	(hl),a
	inc	hl
	ex	de,hl
	jr	pts1
pts2:	pop	af
	ret


;--------------------------------------
; SD command
;
; Parameters:	A: command
;		DE:HL argument (32 bits)
; Returns:	A: R1 state
; Modiffies:	nothing else

sd_cmd:
	rst	$18	;call	SD_CS_L
	push	bc
	ld	b,$87
	or	a
	jr	nz,sdcm1
	ld	b,$95
sdcm1:	or	$40
	rst	8	;call	spibyte	; CMD
	ld	a,d
	rst	8	;call	spibyte	; arg MSB
	ld	a,e
	rst	8	;call	spibyte
	ld	a,h
	rst	8	;call	spibyte
	ld	a,l
	rst	8	;call	spibyte	; arg LSB
	ld	a,b
	rst	8	;call	spibyte	; CRC

	ld	b,0
sdcm2:	ld	a,$FF	; wait response
	rst	8	;call	spibyte
	cp	$FF
	jr	nz,sdcm3
	djnz	sdcm2

sdcm3:	ld	b,a
	ld	a,$ff
	rst	8	;call	spibyte	
	rst	$20	;call	SD_CS_H
	rst	8	;call	spibyte	
	ld	a,b
	pop	bc
	ret
;--------------------------------------
; SD command, 32 bit result
;
; Parameters:	A: command
;		DE:HL argument (32 bits)
; Returns:	A: R1 state
;		DE:HL 32 bit result (sometimes)
; Modiffies:	nothing else

sd_cmd32:
	rst	$18	;call	SD_CS_L
	push	bc
	ld	b,$87
	or	a
	jr	nz,sdc31
	ld	b,$95
sdc31:	or	$40
	rst	8	;call	spibyte	; CMD
	ld	a,d
	rst	8	;call	spibyte	; arg MSB
	ld	a,e
	rst	8	;call	spibyte
	ld	a,h
	rst	8	;call	spibyte
	ld	a,l
	rst	8	;call	spibyte	; arg LSB
	ld	a,b
	rst	8	;call	spibyte	; CRC

	ld	b,0
sdc32:	ld	a,$FF	; wait response
	rst	8	;call	spibyte
	cp	$FF
	jr	nz,sdc33
	djnz	sdc32

sdc33:	ld	b,a
	ld	a,$ff
	rst	8	;call	spibyte	
	ld	d,a
	ld	a,$ff
	rst	8	;call	spibyte	
	ld	e,a
	ld	a,$ff
	rst	8	;call	spibyte	
	ld	h,a
	ld	a,$ff
	rst	8	;call	spibyte	
	ld	l,a
	ld	a,$ff
	rst	8	;call	spibyte	
	rst	$20	;call	SD_CS_H
	ld	a,b
	pop	bc
	ret

;--------------------------------------
; SD init
;
; Parameters:	none
; Returns:	A: 0 OK
; Modiffies:	lots of registers

sd_init:
	in	a,(CONTROL)	; slow clock for SPI
	and	a,~SPIFAST
	out	(CONTROL),a
	
	rst	$20	;call	SD_CS_H
	ld	b,10
sdini1:	rst	8	;call	spibyte
	djnz	sdini1
	rst	$18	;call	SD_CS_L
	ld	b,128	;32
sdini2:	xor	a	; CMD go idle state
	ld	de,0
	ld	hl,0
	call	sd_cmd
	cp	1
	jr	z,sdini3
	djnz	sdini2
	ld	a,1	; Error:timeout
	ret
sdini3: ld	b,0
sdini35:
	ld	a,8	; SEND_IF_COND
	ld	de,0
	ld	hl,$01aa
	call	sd_cmd32
	cp	1
	jr	z,sdini40
	djnz	sdini35
sdini40:
	ld	b,0
sdini4:	ld	a,55	; APP_CMD
	ld	de,0
	ld	hl,0
	call	sd_cmd
	ld	a,41	;SD_SEND_OP_COND,0x40000000
	ld	de,$4000
	call	sd_cmd	
	cp	0
	jr	z,sdini5
	djnz	sdini4
	ld	a,2	; Error card initialization failed
	ret
sdini5:	ld	b,0
sdini6:	ld	a,58	; Read OCR
	ld	de,0
	ld	hl,0
	call	sd_cmd32
	cp	0
	jr	z,sdini9
	djnz	sdini6
	ld	a,3
	ret
sdini9:	in	a,(CONTROL)	; Fast clock for SPI
	or	a,SPIFAST
	out	(CONTROL),a
	ld	a,0	;OK
	ret
	
;--------------------------------------
; SD read block
;
; Parameters:	DE:HL: nº sector
;		IX: puntero
; Returns:	A: 0 OK, !=0 Error
; Modiffies:	IX = IX+512

sd_rd_block:
	push	bc
	ld	a,17	; READ SINGLE BLOCK
	call	sd_cmd
	or	a
	jr	nz,sdrd9
	rst	$18	;call	SD_CS_L
	ld	b,0
sdrd2:	ld	a,$ff
	rst	8	;call	spibyte
	cp	$fe
	jr	z,sdrd3
	djnz 	sdrd2
	ld	a,4
	jr	sdrd9
sdrd3:	ld	b,0
sdrd4:	ld	a,$ff
	out	(SPI),a	;call	spibyte
	in	a,(SPI)
	ld	(ix+0),a
	inc	ix
	ld	a,$ff	;call	spibyte
	out	(SPI),a 
	in	a,(SPI)
	ld	(ix+0),a
	inc	ix
	djnz	sdrd4
	
	ld	a,$ff
	rst	8	;call	spibyte
	ld	a,$ff
	rst	8	;call	spibyte
	ld	a,$ff
	rst	8	;call	spibyte
	
	ld	a,0
sdrd9:	rst	$20	;call	SD_CS_H
	pop	bc
	ret

;--------------------------------------
; SD write block
;
; Parameters:	DE:HL: nº sector
;		IX: puntero
; Returns:	A: 0 OK, !=0 Error
; Modiffies:	IX = IX+512
	
sd_wr_block:
	push	bc
	ld	a,24		; WRITE SINGLE BLOCK
	call	sd_cmd
	or	a
	jr	nz,sdrd9
	rst	$18		;call	SD_CS_L
	ld	a,$FE		;Send start block token 0xfe
	out	(SPI),a
	ld	b,0		; Send data
sdwr4:	ld	a,(ix+0)
	out	(SPI),a	
	inc	ix
	ld	a,(ix+0)
	out	(SPI),a
	inc	ix
	djnz	sdwr4
	
	ld	a,$ff		; send dummy CRC
	rst	8		
	ld	a,$ff
	rst	8

	ld	bc,0		; wait for data response
sdwr5:	ld	a,$ff
	rst	8		
	cp	$ff
	jr	nz,sdwr6
	dec	bc
	ld	a,b
	or	c
	jr	nz,sdwr5
	jr	sdwr91		; timeout
sdwr6:	and	$1F		; AAA='010' - data accepted ;
	cp	5
	jr	nz,sdwr91	; error

	ld	bc,0		;Wait while card busy
sdwr7:	ld	a,$ff
	rst	8
	or	a
	jr	nz,sdwr92
	dec	bc
	ld	a,b
	or	c
	jr	nz,sdwr7
sdwr91:	ld	a,$ff		; timeout
	rst	8
	rst	$20		;call	SD_CS_H
	rst	8
	ld	a,1
	pop	bc
	ret
sdwr92:	ld	a,$ff		; OK
	rst	8
	rst	$20		;call	SD_CS_H
	rst	8
	xor	a
	pop	bc
	ret

;--------------------------------------
;--------------------------------------
; FAT init
; Parameters:	none
; Returns:	A: 0 OK, !=0 Error
; Modiffies:	lots

FAT_init:
	ld	iy,fatbas
	ld	ix,fatbuf
	ld	de,0
	ld	hl,0
	call	sd_rd_block	; tabla de particiones
	cp	0
	ret	nz
	ld	de,-2
	add	ix,de	
	ld	a,$55
	cp	(ix+0)
	ret	nz
	ld	a,$AA
	cp	(ix+1)
	ret	nz
	ld	de,-56
	add	ix,de
	ld	d,(ix+3)
	ld	e,(ix+2)
	ld	h,(ix+1)
	ld	l,(ix+0)
	
	ld	ix,fatbuf
	call	sd_rd_block	; superblock
	cp	0
	ret	nz
	
	ld	ix,fatbuf
	ld	b,(ix+$d)	; Sectors/Cluster
	ld	a,b
	ld	(spc),a
	ld	a,0
fini1:	inc	a
	srl	b
	jr	nz,fini1
	dec	a
	ld	(l2spc),a	; log2(sector_per_cluster)

	ld	b,(ix+$f)	; BC=nº de sectores reservados
	ld	c,(ix+$e)
	add	hl,bc
	jr	nc,fini2
	inc	e
	jr	nc,fini2
	inc	d
fini2:	ld	(iy+fat1_begin-fatbas),l
	ld	(iy+fat1_begin-fatbas+1),h
	ld	(iy+fat1_begin-fatbas+2),e
	ld	(iy+fat1_begin-fatbas+3),d

	ld	b,(ix+$10)	; nº de FATs
fini3:	ld	a,(ix+$24)	; ($24-$27): sectores/FAT
	add	l
	ld	l,a
	ld	a,(ix+$25)	
	adc	h
	ld	h,a
	ld	a,(ix+$26)	
	adc	e
	ld	e,a
	ld	a,(ix+$27)	
	adc	d
	ld	e,a
	djnz	fini3
	ld	(iy+storage_begin-fatbas),l	; storage begin
	ld	(iy+storage_begin-fatbas+1),h
	ld	(iy+storage_begin-fatbas+2),e
	ld	(iy+storage_begin-fatbas+3),d
	
	ld	l,(ix+$2c)	; root_cluster
	ld	h,(ix+$2d)
	ld	e,(ix+$2e)
	ld	d,(ix+$2f)
	ld	(iy+root_cluster-fatbas),l
	ld	(iy+root_cluster-fatbas+1),h
	ld	(iy+root_cluster-fatbas+2),e
	ld	(iy+root_cluster-fatbas+3),d
	
	ld	a,$ff		; last_cluster
	ld	(iy+last_cluster-fatbas),a
	ld	(iy+last_cluster-fatbas+1),a
	ld	(iy+last_cluster-fatbas+2),a
	ld	(iy+last_cluster-fatbas+3),a

	ld	a,0
	ret
;--------------------------------------
;--------------------------------------
; next_cluster
; Parameters:	DE:HL : Cluster actual
; Returns:	DE:HL : siguiente cluster
; Modiffies:	A

next_cluster:
	push	bc
	push	ix
	push	iy
	ld	iy,fatbas
	ld	a,$7F
	and	l
	ld	c,a	; c=cluster&127
	ld	a,l
	xor	(iy+last_cluster-fatbas)
	and	$80
	jr	nz,nxc1
	ld	a,h
	cp	(iy+last_cluster-fatbas+1)
	jr	nz,nxc1
	ld	a,e
	cp	(iy+last_cluster-fatbas+2)
	jr	nz,nxc1
	ld	a,d
	cp	(iy+last_cluster-fatbas+3)
	jr	z,nxc2
nxc1:	ld	(iy+last_cluster-fatbas),l	; last_cluster=cluster
	ld	(iy+last_cluster-fatbas+1),h
	ld	(iy+last_cluster-fatbas+2),e
	ld	(iy+last_cluster-fatbas+3),d
	;cluster>>7
	rl	l
	rl	h
	ld	l,h
	rl	e
	ld	h,e
	rl	d
	ld	e,d
	ld	a,0
	rl	a
	ld	d,a
	; +fat_1_begin
	ld	a,l
	add	(iy+fat1_begin-fatbas)
	ld	l,a
	ld	a,h
	add	(iy+fat1_begin-fatbas+1)
	ld	h,a
	ld	a,e
	add	(iy+fat1_begin-fatbas+2)
	ld	e,a
	ld	a,d
	add	(iy+fat1_begin-fatbas+3)
	ld	d,a
	ld	ix,fatbuf
	call	sd_rd_block
nxc2:	ld	b,0	; BC*4
	sla	c
	rl	b
	sla	c
	rl	b
	ld	ix,fatbuf
	add	ix,bc
	ld	l,(ix+0)
	ld	h,(ix+1)
	ld	e,(ix+2)
	ld	d,(ix+3)
	pop	iy
	pop	ix
	pop	bc
	ret
;--------------------------------------
;--------------------------------------
; wrfile
; Parameters:	IX : buffer
; Returns:	BC : 512 (unless EOF)
;		IX = IX+nbytes
; Modiffies:	A
wrfile:	ld	a,$ff
	jr	rdf0
;--------------------------------------
;--------------------------------------
; rdfile
; Parameters:	IX : buffer
; Returns:	BC : read bytes (512 unless EOF)
;		IX = IX+nbytes
; Modiffies:	A

rdfile:	xor	a
rdf0:	ld	(rdwr),a
	push	de
	push	hl
	push	iy
	ld	iy,fatbas
	ld	a,(iy+filesector-fatbas)	; A,C = ns = filesector+1
	inc	a
	ld	c,a
	ld	l,(iy+filecluster-fatbas)	; DE:HL = nc =filecluster
	ld	h,(iy+filecluster-fatbas+1)
	ld	e,(iy+filecluster-fatbas+2)
	ld	d,(iy+filecluster-fatbas+3)
	cp	(iy+spc-fatbas)			; if (ns==SecPerCluster)
	jr	nz,rdf1
	ld	c,0				; ns = 0
	call	next_cluster
rdf1:	ld	a,(iy+filecluster-fatbas+3)	; if (filecluster>=0x0FffffF0)
	or	$F0
	and	(iy+filecluster-fatbas+2)
	and	(iy+filecluster-fatbas+1)
	cp	$FF
	jr	nz,rdf15
	ld	a,$0F
	or	(iy+filecluster-fatbas+0)
	cp	$FF
	jr	nz,rdf15
rdf91:	ld	bc,0				; return 0
	jp	rdf99
rdf15:	ld	a,fatbuf&$ff			; if (IX==fatbuf)
	cp	ixl
	jr	nz,rdf2
	ld	a,fatbuf/256
	cp	ixh
	jr	nz,rdf2
	ld	a,$ff
	ld	(iy+last_cluster-fatbas),a	; lastcluster=0xFFFFFFFF
	ld	(iy+last_cluster-fatbas+1),a
	ld	(iy+last_cluster-fatbas+2),a
	ld	(iy+last_cluster-fatbas+3),a
rdf2:	push	de	; sector=storage_begin+((filecluster-2)<<l2spc)+filesector
	push 	hl
	ld	a,(iy+filecluster-fatbas)
	sub	2
	ld	l,a
	ld	a,(iy+filecluster-fatbas+1)
	sbc	0
	ld	h,a
	ld	a,(iy+filecluster-fatbas+2)
	sbc	0
	ld	e,a
	ld	a,(iy+filecluster-fatbas+3)
	sbc	0
	ld	d,a
	ld	b,(iy+l2spc-fatbas)
rdf3:	sla	l
	rl	h
	rl	e
	rl	d
	djnz	rdf3
	ld	a,l
	add	(iy+storage_begin-fatbas)	
	ld	l,a
	ld	a,h
	adc	(iy+storage_begin-fatbas+1)
	ld	h,a
	ld	a,e
	adc	(iy+storage_begin-fatbas+2)
	ld	e,a
	ld	a,d
	adc	(iy+storage_begin-fatbas+3)
	ld	d,a
	ld	a,(iy+filesector-fatbas)
	add	l
	ld	l,a
	ld	a,0
	adc	h
	ld	h,a
	ld	a,0
	adc	e
	ld	e,a
	ld	a,0
	adc	d
	ld	d,a
	ld	a,(rdwr)	; read or write?
	or	a
	jr	nz,rdf5
	call	sd_rd_block
	jr	rdf51
rdf5:	call	sd_wr_block
rdf51:	pop	hl		; DE:HL = nc again
	pop	de
	or	a
	jp	nz,rdf91
	ld	(iy+filesector-fatbas),c	; filesector=ns
	ld	(iy+filecluster-fatbas),l	; filecluster=nc
	ld	(iy+filecluster-fatbas+1),h
	ld	(iy+filecluster-fatbas+2),e
	ld	(iy+filecluster-fatbas+3),d
	
	ld	bc,512				; if (filelen>=512) bc=512
	ld	a,$FE
	and	(iy+filelen-fatbas+1)
	or	(iy+filelen-fatbas+2)
	or	(iy+filelen-fatbas+3)
	jr	nz,rdf4
	ld	c,(iy+filelen-fatbas)		; else BC=filelen
	ld	b,(iy+filelen-fatbas+1)
rdf4:	ld	a,(iy+filelen-fatbas)		; filelen-=BC
	sub	c
	ld	(iy+filelen-fatbas),a
	ld	a,(iy+filelen-fatbas+1)
	sbc	b
	ld	(iy+filelen-fatbas+1),a
	ld	a,(iy+filelen-fatbas+2)
	sbc	0
	ld	(iy+filelen-fatbas+2),a
	ld	a,(iy+filelen-fatbas+3)
	sbc	0
	ld	(iy+filelen-fatbas+3),a
	ld	a,0
rdf99:	pop	iy			; return BC
	pop	hl
	pop	de
	ret

;--------------------------------------
;--------------------------------------
; fstream
; Parameters:	none
; Returns:	A = next byte from file
; Modiffies:	IX

fstream:
	push	de
	push	hl
	ld	de,(bufix)
	ld	hl,(bufmax)
	
	ld	a,d
	xor	h
	jr	nz,fst1
	ld	a,e
	xor	l
	jr	nz,fst1
	push	bc
	ld	ix,fatbuf
	call	rdfile
	ld	(bufmax),bc
	pop	bc
	ld	de,0
	ld	(bufix),de
fst1:	ld	hl,fatbuf
	add	hl,de
	ld	a,(hl)
	inc	de
	ld	(bufix),de
	
	pop	hl
	pop	de
	ret
;--------------------------------------
;--------------------------------------
; feof
; Parameters:	none
; Returns:	A !=0 if end of file
; 
feof:	push	de
	push	hl
	ld	de,(bufix)
	ld	hl,(bufmax)
	ld	a,h
	sub	2
	jr	z,feof9
	ld	a,l
	cp	e
	jr	nz,feof8
	ld	a,h
	cp	d
	jr	nz,feof8
	ld	a,$FF
	jr	feof9
feof8:	xor	a	
feof9:	pop	hl
	pop	de
	ret	


;-------------------------
; File selector
;-------------------------
filesel:
	
	call	frameit
fs000:	call	init
	or	a
	jr	nz,fs000
	call	frameit
	; start at root directory
	ld	de,curdir
	ld	hl,root_cluster
	ld	bc,4
	ldir

fs00:	ld	iy,fselbas	; fsel=pos=ppos=offset=0
	xor	a
	ld	(iy+fsel-fselbas),a
	ld	(iy+pos-fselbas),a
	ld	(iy+ppos-fselbas),a
	ld	(iy+ppos-fselbas+1),a
	ld	(iy+offset-fselbas),a
	ld	(iy+offset-fselbas+1),a

fs0:	; read current directory
	ld	de,filecluster
	ld	hl,curdir
	ld	bc,4
	ldir
	
	xor	a
	ld	(filesector),a
	ld	a,$ff
	ld	(filelen+3),a
	
	;------------------

fs1:	ld	ix,fatbuf	; scan directory entries
	call	rdfile	
	ld	ix,fatbuf
	ld	b,16
fs2:	ld	a,[ix+0]
	or	a
	jp	z,fs99
	cp	$E5		; deleted
	jp	z,fsskip
	ld	a,[ix+11]
	and	$30		; File or Dir
	jp	z,fsskip
	; File OK: count it
	ld	a,(iy+ppos-fselbas)
	sub	(iy+offset-fselbas)
	ld	a,(iy+ppos-fselbas+1)
	sbc	(iy+offset-fselbas+1)
	jr	z,fss1
	inc	(iy+ppos-fselbas)
	jr	nz,fsskip
	inc	(iy+ppos-fselbas+1)
	jr	fsskip
fss1:	; copy file metadata
	ld	l,(iy+pos-fselbas)
	ld	h,0
	add	hl,hl	; hl*=32
	add	hl,hl
	add	hl,hl
	add	hl,hl
	add	hl,hl
	ld	de,filedata
	add	hl,de
	ex	de,hl
	push	ix
	pop	hl
	push	bc
	ld	bc,32
	ldir
	pop	bc
	; print name, register file, register file

	ld	a,[ix+11] ; save file attributes
	push	af
	xor	a	; Terminate string
	ld	[ix+11],a
	ld	a,[ix+8]
	push	af	; save first Ext chararter
	xor	a	; Terminate string
	ld	[ix+8],a
	push	ix
	pop	hl
	call	pos2video
	call	puts	; Print name
	pop	af
	ld	[ix+8],a
	ex 	de,hl
	ld	a,32
	ld	(hl),a
	ex	de,hl
	inc	e
	call	puts	; Print Extension
	pop	af
	and	$10	; Print 'D' if subdir
	ld	hl,txtfil
	jr	z,fs4
	ld	hl,txtdir
fs4:	call	puts
	ld	a,(iy+fsel-fselbas)
	cp	(iy+pos-fselbas)
	jr	nz,fss2
	call	pos2attr
	ex	de,hl
	push	bc
	ld	bc,$0E80 ; Remark
fs25:	ld	a,(hl)
	or	c
	ld	(hl),a
	inc	hl
	djnz	fs25
	pop	bc

fss2:
	inc	(iy+pos-fselbas)
	ld	a,32
	cp	(iy+pos-fselbas)
	jr	z,fs99
fsskip:			; Next file entry
	ld	de,32
	add	ix,de
	dec	b
	jp	nz,fs2
	
	jp	fs1	; Read more sectors of directory

	;-------------- Wait for KEY action -------------

fs99:	; clear until the end of screen
	ld	a,(iy+pos-fselbas)
	push	af
fs9900:	ld	a,32
	cp	(iy+pos-fselbas)
	jr	z,fs9903
	call	clrpos
	inc	(iy+pos-fselbas)
	jr	fs9900
fs9903:	pop	af
	ld	(iy+pos-fselbas),a

	call	getkey		; Wait for key
	cp	$74 ;'P'		; Right move
	jr	nz,fs991
	ld	a,(iy+fsel-fselbas)
	add	16
	cp	(iy+pos-fselbas)
	jr	nc,fs9902
	ld	(iy+fsel-fselbas),a
	jr	fs9901
fs9902:	ld	a,32
	cp	(iy+pos-fselbas)
	jr	nz,fs9901
	ld	l,(iy+offset-fselbas)
	ld	h,(iy+offset-fselbas+1)
	ld	de,16
	add	hl,de
	ld	(iy+offset-fselbas),l
	ld	(iy+offset-fselbas+1),h
fs9901:	jp	fs999
	
fs991:	cp	$6B ;'O'		; Left move
	jr	nz,fs992
	ld	a,$f0
	and	(iy+fsel-fselbas)
	jr	z,fs9912
	ld	a,(iy+fsel-fselbas)
	sub	16
	ld	(iy+fsel-fselbas),a
	jr	fs9911
fs9912:	ld	a,$F0
	and	(iy+offset-fselbas)
	or	(iy+offset-fselbas+1)
	jr	z,fs9911
	ld	l,(iy+offset-fselbas)
	ld	h,(iy+offset-fselbas+1)
	ld	de,$10000-16
	add	hl,de
	ld	(iy+offset-fselbas),l
	ld	(iy+offset-fselbas+1),h
fs9911:	jp	fs999
	
fs992:	cp	$72 ;'A'		; Down move
	jr	nz,fs993
	inc	(iy+fsel-fselbas)
	ld	a,(iy+fsel-fselbas)
	cp	(iy+pos-fselbas)
	jr	c,fs9921
	ld	a,(iy+pos-fselbas)
	dec	a
	ld	(iy+fsel-fselbas),a
fs9921:	jp	fs999
	
fs993:	cp	$75 ;'Q'		; Up move
	jr	nz,fs994
	ld	a,(iy+fsel-fselbas)
	or	a
	jr	z,fs9931
	dec	(iy+fsel-fselbas)
fs9931:	jp	fs999
	
fs994:	;cp	'B'		; Basic
	;jp	z,gorom
	cp	$7c		; (pet sys) OFF display
	jr	nz,fs9940
fsoff:	in	a,(CONTROL)
	and	~GRAB
	out	(CONTROL),a
	call	waitaction
	call	frameit
	jp	fs999
	
fs9940:	cp	$5A ; 10		; Enter or Space
	jr	z,fs9941
	cp	$29 ;32
	jr	z,fs9941
	cp	$2D	; 'R'
	jp	nz,fs998
	jr	fs9942
fs9941:	xor	a
fs9942:	ld	(rlewr),a
	ld	l,(iy+fsel-fselbas)
	ld	h,0
	add	hl,hl	; hl*=32
	add	hl,hl
	add	hl,hl
	add	hl,hl
	add	hl,hl
	ld	de,filedata
	add	hl,de
	push	hl
	pop	ix
	ld	a,$10	; Subdir?
	and	(ix+11)
	jr	z,fs995
	ld	l,(ix+26)
	ld	h,(ix+27)
	ld	e,(ix+20)
	ld	d,(ix+21)
	ld	a,l	; cluster==0 => root
	or	h
	or	e
	or	d
	jp	z,filesel
	ld	ix,curdir
	ld	(ix+0),l
	ld	(ix+1),h
	ld	(ix+2),e
	ld	(ix+3),d
	ld	a,$ff
	ld	(filelen),a
	jp	fs00

fs995:	ld	a,'R'	; RLE
	cp	(ix+8)
	jp	nz,fs996
	ld	a,'L'
	cp	(ix+9)
	jp	nz,fs996
	ld	a,'E'
	cp	(ix+10)
	jp	nz,fs996
	ld	l,(ix+26)
	ld	h,(ix+27)
	ld	e,(ix+20)
	ld	d,(ix+21)
	ld	(filecluster),hl
	ld	(filecluster+2),de
	ld	l,(ix+28)
	ld	h,(ix+29)
	ld	e,(ix+30)
	ld	d,(ix+31)
	ld	(filelen),hl
	ld	(filelen+2),de
	xor	a
	ld	(filesector),a
	
	ld	a,(rlewr)
	or	a
	jp	nz,rlw0
	;;;;;;;;;;;;; RLE player ;;;;;;;;;;
	
	call	cls
	in	a,(CONTROL)
	and	~GRAB
	out	(CONTROL),a
	ld	de,video
	ld	hl,textorle
	call	puts
	ld	de,video+32
	ld	hl,textorle2
	call	puts

	ld	ix,buffer0
	ld	(pbuf),ix
	call	rdfile
	call	rdfile
	ei
	; 1st buffer
rlepl1:	; Check for abort key (sysrq)
	call	chksrq
	cp	$7c
	jr	z,rlepl92
rlepl3:	ld	a,(pbuf+1)
	and	$FE
	cp	$18
	jr	z,rlepl1
	ld	ix,buffer0
	call	rdfile
	ld	a,2
	cp	b
	jr	nz,rlepl9
	
	call	prtkb	; Print remaining KBs

	; 2nd buffer		
rlepl2:	; Check for abort key (sysrq)
	call	chksrq
	cp	$7c
	jr	z,rlepl92
	
rlepl4:	ld	a,(pbuf+1)
	and	$FE
	cp	$1A
	jr	z,rlepl2
	;ld	ix,buffer1
	call	rdfile
	ld	a,2
	cp	b
	jr	z,rlepl1
	
rlepl9:	push	ix
	pop	hl
	add	hl,bc
	
rlepl91: 
	in	a,(STATUS)
	and	MOTOR
	jr	z,rlepl92
	ld	a,(pbuf)
	cp	l
	jr	nz,rlepl91
	ld	a,(pbuf+1)
	cp	h
	jr	nz,rlepl91
rlepl92:
	di
	jp	fsoff
textorle:	db 127,"Playing RLE:      KB remaining",127,0
textorle2:	db 127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127
		db 127,127,127,127,127,127,127,127,127,127,127,127,0
textorle3:	db 127,"Recording RLE:      KB        ",127,0
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
rlw0:	;;;;;;;;;; RLE recording ;;;;;;;;;;;;;
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	xor	a
	call	cls
	
	ld	de,video
	ld	hl,textorle3
	call	puts
	ld	de,video+32
	ld	hl,textorle2
	call	puts
	; wait for motor ON
	in	a,(CONTROL)
	and	~GRAB
	out	(CONTROL),a
rlew1:	in	a,(STATUS)
	and	MOTOR
	jr	z,rlew1	
	
	ld	a,127
	out	(RLE),a
	ld	ix,buffer0
	ld	(pbuf),ix
	ld	de,0
	ei
	
rlew2:	call	chksrq
	cp	$7c	; sys req?
	jr	z,rlew9
	ld	a,(pbuf+1)
	and	$FE
	cp	$18
	jr	z,rlew2
	ld	ix,buffer0
	call	wrfile

rlew3:	call	chksrq
	cp	$7c	; sys req?
	jr	z,rlew9
	ld	a,(pbuf+1)
	and	$FE
	cp	$1A
	jr	z,rlew3
	ld	ix,buffer1
	call	wrfile
	inc	de
	push	de
	ld	hl,video+15
	call	prtdec16
	pop	de
	jr	rlew2

rlew9:	di
	jp	fsoff

chksrq:	in	a,(STATUS)
	and	KEYVAL
	ret	z
	in	a,(KEYB)
	ret
	
fs996:	ld	a,'C'	; Amstrad TZX (.CDT)
	cp	(ix+8)
	jr	nz,fs997
	ld	a,'D'
	cp	(ix+9)
	jr	nz,fs997
	ld	a,'T'
	cp	(ix+10)
	jr	nz,fs997
fs9960:	ld	l,(ix+26)
	ld	h,(ix+27)
	ld	e,(ix+20)
	ld	d,(ix+21)
	ld	(filecluster),hl
	ld	(filecluster+2),de
	ld	l,(ix+28)
	ld	h,(ix+29)
	ld	e,(ix+30)
	ld	d,(ix+31)
	ld	(filelen),hl
	ld	(filelen+2),de
	xor	a
	ld	(filesector),a
	ld	bc,0
	ld	(bufix),bc
	ld	(bufmax),bc
	jp	playTZX 

fs997:	ld	a,'T'	; .TZX file (same as .CDT)
	cp	(ix+8)
	jp	nz,fs998
	ld	a,'Z'
	cp	(ix+9)
	jp	nz,fs998
	ld	a,'X'
	cp	(ix+10)
	jp	nz,fs998
	jr	fs9960
	
fs998:	;cp	'H'		; Hex dump (debug)
	;jr	nz,fs999
	;jp	fs0
	
fs999:	xor	a
	ld	(iy+pos-fselbas),a
	ld	(iy+ppos-fselbas),a
	ld	(iy+ppos-fselbas+1),a
	jp	fs0

;;;;;;;;;;;;;;;;;;; some routines ;;;;;;;;;;;;;;;;;;;

pos2video:	; DE = starting screen address for current position
pos2attr:
	ld	de,$1CE1
	ld	a,(pos)
	and	16
	jr	z,p2v2
	ld	e,$F1
p2v2:	ld	a,(pos)
	and	15
	sla	a
	sla	a
	sla	a
	sla	a
	sla	a
	jr	nc,p2v3
	inc	d
p2v3:	add	e
	ld	e,a
	ret	nc
	inc	d
	ret
	
clrpos:	push	bc	; clear 16-char line at (pos)
	push	de
	push	hl
	call	pos2video	; video memory
	ld	h,d
	ld	l,e
	ld	a,32
	ld	(hl),a
	inc	de
	ld	bc,13
	ldir
	pop	hl
	pop	de
	pop	bc
	ret

; draw screen frame 
frameit:
	ld	a,32
	call	cls
	ld	hl,video
	ld	de,video+1
	ld	a,127
	ld	(hl),a
	ld	bc,31
	ldir
	ld	hl,video+23*32
	ld	de,video+23*32+1
	ld	(hl),a
	ld	bc,31
	ldir
	ld	b,22
	ld	hl,video+32
	ld	de,31
frit1:	ld	(hl),a
	add	hl,de
	ld	(hl),a
	inc	hl
	djnz	frit1
	; print info
	ld	de,video+64+2
	ld	hl,txto1
	call	puts
	ld	de,video+128+1
	ld	hl,txto4
	call	puts

	ret
	
; Wait for key pressed 
	
getkeysc:
gksc1:	in	a,(STATUS)
	and	KEYVAL
	jr	z,gksc1
	in	a,(KEYB)
	ret

gk0:	call	getkeysc
getkey:	call	getkeysc
	cp	$F0	; // filter released keys out
	jr	z,gk0
	cp	$E0	; // filter extended scancodes out
	jr	z,getkey
	ret

; AUX printing routines
	
prthex:	push	af
	srl	a	
	srl	a	
	srl	a	
	srl	a	
	cp	10
	jr	c,prth1
	add	7
prth1:	add	48
	ld	(hl),a
	inc	hl
	pop	af
	and	15
	cp	10
	jr	c,prth2
	add	7
prth2:	add	48
	ld	(hl),a
	inc	hl
	ret

; 16 bit print DE in dec
;	

prtdec16:
	push	bc
	ld	b,6
	ld	a,32
pde1:	ld	(hl),a
	inc	hl
	djnz	pde1
pde4:	ld	b,16
	xor	a	
pde2:	sla	e
	rl	d
	rl	a
	cp	10
	jr	c,pde3
	sub	10
	inc	e
pde3:	djnz	pde2	; A remainder
	add	48
	dec	hl
	ld	(hl),a
	ld	a,e
	or	d
	jr	nz,pde4	
	pop	bc
	ret	
	
txto1:	db "TAPE80 ROM   J. Arias (2022)",0
txto4:	db "SD-HC FAT32 tape File SELECTOR",0
txtdir:	db " D",0
txtfil:	db "  ",0

;----------------------------------------------------
; convert T-states (3.5MHz cycles) to samples (44KHz)
; A = DE/64 - DE/512 - DE/1024
; Input: 	DE: tstates
; return: 	A: samples
; modiffies:	DE
tst2sam:
	ld	a,d
	rl	e
	rla
	rl	e
	rla
	srl	d
	sub	d
	srl	d
	sub	d
	ret

;----------------------------------------------------
; Put RLE data in the buffer
; Input: 	A: data to put
; modiffies: 	None

putrle:	push	de
	push	hl
	push	af
	ld	de,(pmbuf)	; increment pointer (circular)
	inc	de
	ld	a,d
	cp	$1C
	jr	nz,prl1
	ld	d,$18
prl1:	ld	hl,(pbuf)	; (++pmbuf==pbuf) ?
	ld	a,l
	cp	e
	jr	nz,prl2
	ld	a,h
	cp	d
	jr	nz,prl2
	ei			; buffer full: enable INTs, just in case...
	call	chksrq		; and check keyboard abort
	cp	$7c		; Abort <sysreq>
	jp	nz,prl1
	pop	af
	pop	hl
	pop	de
	pop	af		; return address out of stack
	jp	rlew9

prl2:	pop	af
	ld	hl,(pmbuf)
	ld	(hl),a		; write to buffer
	ld	(pmbuf),de	; update pointer
	pop	hl
	pop	de
	ret

;-------------------------------------------------------
;   TZX player
;-------------------------------------------------------
textotzx:	db 127,"Playing TZX:      KB remaining",127,0

playTZX:
	call	cls
	in	a,(CONTROL)
	and	~GRAB
	out	(CONTROL),a
	ld	de,video
	ld	hl,textotzx
	call	puts
	ld	de,video+32
	ld	hl,textorle2
	call	puts
	call	prtkb

	call	fstream
	cp	'Z'
	jp	nz,ptz99
	call	fstream
	cp	'X'
	jp	nz,ptz99
	call	fstream
	cp	'T'
	jp	nz,ptz99
	ld	hl,buffer0	; Initialize pointers
	ld	(pbuf),hl
	ld	(pmbuf),hl
	ld	b,7
ptz01:	call	fstream
	djnz	ptz01
ptz1:	call	fstream		; Block ID
	ld	b,a
	call	feof
	or	a
	jp	nz,ptz99
	ld	a,b
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	; Text Description
	cp	$30		
	jr	nz,ptz10
ptz010:	call	fstream		; number of bytes
	ld	b,a	
	jr	ptz01
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	; Pause
ptz10:	cp	$20		
	jr	nz,ptz11
	call	fstream		; number of ms
	ld	c,a
	call	fstream
	ld	b,a
ptz101:	ld	a,44		; 44 samples = 1ms
	call	putrle
	dec	bc
	ld	a,b
	or	c
	jr	nz,ptz101	
	jr	ptz1
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	; Turbo speed data block
ptz11:	cp	$11		
	jp	nz,ptz12
	ld	b,5
	ld	hl,turboblk
ptz110:	call	fstream		; T-states
	ld	e,a
	call	fstream
	ld	d,a
	call	tst2sam		; to samples
	ld	(hl),a		; store in data structure
	inc	hl
	djnz	ptz110
	ld	b,5
ptz111:	call	fstream
	ld	(hl),a		; store in data structure
	inc	hl
	djnz	ptz111
	call	fstream		; LDE: 24 bit counter
	ld	e,a
	call	fstream
	ld	d,a
	call	fstream
	ld	l,a
	; Pilot tone
	ld	bc,(npilot)	; npulses/2
	srl	b
	rr	c
ptz112:	ld	a,(sampilot)
	or	$80
	call	putrle
	xor	$80
	call	putrle
	dec	bc
	ld	a,b
	or	c
	jr	nz,ptz112	
	; Sync
	ld	a,(samsyn1)
	or	$80
	call	putrle
	ld	a,(samsyn2)
	call	putrle
	; data
ptz113:	call	fstream
ptz1131:
	ld	h,a
	ld	b,8
ptz114:	ld	a,(samzero)
	rl	h
	jr	nc,ptz115
	ld	a,(samone)
ptz115: or	$80
	call	putrle
	xor	$80
	call	putrle
	djnz	ptz114
	ld	a,e
	or	a
	call	z,prtkb	; print KB remaining
	ld	a,e	; LDE--
	or	d
	jr	nz,ptz116
	dec	l
ptz116:	dec	de
	ld	a,e
	or	d
	or	l
	jr	nz,ptz113
	; Pause	
	ld	bc,(mspause)
	jp	ptz101
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	; Pure tone
ptz12:	cp	$12
	jr	nz,ptz13
	call	fstream
	ld	e,a
	call	fstream
	ld	d,a
	call	tst2sam
	ld	h,a
	call	fstream
	ld	c,a
	call	fstream
	ld	b,a
ptz121:	ld	a,h
	xor	$80
	ld	h,a
	call	putrle
	dec	bc
	ld	a,b
	or	c
	jr	nz,ptz121
	jp	ptz1
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	; Pulse sequence
ptz13:	cp	$13
	jr	nz,ptz14
	call	fstream
	ld	b,a
	ld	h,$80
ptz131:	call	fstream
	ld	e,a
	call	fstream
	ld	d,a
	call	tst2sam
	or	h
	call	putrle
	ld	a,h
	xor	$80
	ld	h,a
	djnz	ptz131
	jp	ptz1
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	; Pure data
ptz14:	cp	$14
	jr	nz,ptz15
	call	fstream
	ld	e,a
	call	fstream
	ld	d,a
	call	tst2sam
	ld	(samzero),a
	call	fstream
	ld	e,a
	call	fstream
	ld	d,a
	call	tst2sam
	ld	(samone),a
	call	fstream
	ld	(nbitlast),a
	call	fstream
	ld	(mspause),a
	call	fstream
	ld	(mspause+1),a
	call	fstream
	ld	e,a
	call	fstream
	ld	d,a
	call	fstream
	ld	l,a
	jp	ptz113	; Data section of Turbo block
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	; GROUP start /END
ptz15:	cp	$21
	jp	z,ptz010	; Start: same as Text string
	cp	$22
	jp	z,ptz1		; End: just continue
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	; Standard block (.TAP timmings)
	cp	$10
	jr	nz,ptz16
	call	fstream
	ld	(mspause),a
	call	fstream
	ld	(mspause+1),a
	call	fstream
	ld	e,a
	call	fstream
	ld	d,a
	ld	l,0
	ld	a,11
	ld	(samzero),a	; 10.85 samples
	ld	a,22
	ld	(samone),a	; 21.70 samples
	call	fstream		; first data byte
	ld	h,a
	ld	bc,8062/2	; pilot lenght
	and	$80
	jr	z,ptz151
	ld	bc,3222/2
ptz151:	ld	a,28+128	; pilot
	call	putrle
	ld	a,28
	call	putrle
	dec	bc
	ld	a,b
	or	c
	jr	nz,ptz151
	ld	a,8+128		; sync
	call	putrle
	ld	a,9
	call	putrle
	ld	a,h
	jp	ptz1131	; Go to data section of Turbo speed data block
			; but don't read first byte
ptz16:	cp	$31	; Message block
	jr	nz,ptz17
	call	fstream	; time in seconds
	jp	ptz010	; and now like ID_$30
ptz17:	cp	$32	; Archive info
	jr	nz,ptz18
	call	fstream
	ld	c,a
	call	fstream
	ld	b,a
ptz171:	call	fstream
	dec	bc
	ld	a,b
	or	c
	jr	nz,ptz171
	jp	ptz1
	
ptz18:	cp	$33	; Hardware info
	jr	nz,ptz19
	call	fstream
	ld	c,a	; BC = A*3
	ld	b,0
	sla	c
	rl	b
	add	c
	ld	c,a
	ld	a,b
	adc	0
	ld	b,a
	jr	ptz171	
	
	;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	; Other not yet supported blocks
textonsup:	db "Block: $     not yet supported",0
ptz19:	ld	hl,textonsup
	ld	de,video+1
	call	puts
	ld	hl,video+9
	call	prthex
	;di

	; End of file
ptz99:	call	chksrq	
	cp	$7c		; Abort
	jr	z,ptz991
	ld	hl,(pbuf)	; wait until buffer empty:
	ld	de,(pmbuf)	; pbuf==pmbuf
	ccf	
	sbc	hl,de
	ld	a,h
	or	l
	jr	nz,ptz99
ptz991:	di
	jp	fsoff
	
prtkb:	; Print remaining KBs
	push	de
	push	hl
	ld	hl,video+12
	ld	de,(filelen+1)
	ld	a,(filelen+3)
	srl	a
	rr	d
	rr	e
	srl	a
	rr	d
	rr	e
	call	prtdec16
	pop	hl
	pop	de
	ret
	
;-------------------------------------------------------
;-------------------------------------------------------
;          VARIABLES
;-------------------------------------------------------
;-------------------------------------------------------
rdwr:	db	0		; 1=write for rdfile / wrfile
rlewr:	db	0		; 1=recordong 0=playing (for interrupt)

fatbas:				; FAT variables
storage_begin:	db 0,0,0,0
fat1_begin:	db 0,0,0,0
root_cluster:	db 0,0,0,0
last_cluster:	db 0,0,0,0
l2spc:		db 0		; log2(sectors_per_cluster)
spc:		db 0		; sectors_per_cluster

				; File variables
filelen:	db 0,0,0,0
filecluster:	db 0,0,0,0
filesector:	db 0
bufix:		db 0,0		; for file streaming
bufmax:		db 0,0

; File Selector variables

fselbas:	
fsel:		db 0	; index of selection (<32)
pos:		db 0	; file position after offset
ppos:		db 0,0	; file position before offset
offset:		db 0,0	; first file displayed
curdir:		db 0,0,0,0 ; current directory cluster
pbuf:		db 0,0	; Pointer to buffer (interrupt)
pmbuf:		db 0,0	; pointer to buffer (main)
ostat:		db 0	; last STATUS value (for edge detection of MOTOR)

; TZX variables
turboblk:
sampilot:	db 0	; Samples for pilot semipulse
samsyn1:	db 0	; Samples for SYNC high
samsyn2:	db 0	; Samples for SYNC low
samzero:	db 0	; Samples for Zero semipulse
samone:		db 0	; Samples for One semipulse
npilot:		db 0,0	; Number of pilot pulses
nbitlast:	db 0	; Number of bits in last byte
mspause:	db 0,0	; Number of milliseconds of pause after block


; Buffers
fatbuf:		;block 512 ; part of the File Allocation Table
;filedata:	block 32*32 ; copy of "decent" files (moved to audio buffer)

; 1K AUDIO buffer (also used for directory entries)
	org	6*1024
filedata:	
buffer:
buffer0:
	org	6*1024+512
buffer1:
; Video memory
	org	7*1024
video:
	org	7*1024+768	
endvideo:

