//--------------------------------------------------------------------
// System CPC464
//--------------------------------------------------------------------

module Tapeplayer (
	input clk,		// 25MHz
	input reset,
	
	// VIDEO
	input hsyn,		// VGA sync input
	input vsyn,		// VGA sync input
	output video,	// 1-bit video out
	output rem,		// video remark
	output viden,	// video enable (with blank)
	// SPI
	input	miso,
	output	sck,
	output	mosi,
	output	cs1b,	// Flash
	output	cs2b,	// SD card
	// PS2 KEYBOARD
	input	kclk,
	input	kdat,
	output 	grab,	// Take control of keyboard
	// tape
	input  ck44k,	// CPU_CLK/91 (44.1kHz for normal speed)
	output tapeout,	
	input  tapein,
	input  motor,
	// for scope probes
	output [1:0]debug	
);


/////////////////////////////// CPU ////////////////////////////////

wire reset_n,wait_n,int_n,nmi_n,busrq_n,cen;
wire m1_n,mreq_n,iorq_n,rd_n,wr_n,rfsh_n,halt_n,busak_n;
wire [7:0]cdi;	// CPU Data bus, Input
wire [7:0]cdo;	// CPU Data bus, Output
wire [15:0]ca;	// CPU Address bus

// always inactive signals
assign wait_n=1'b1;
assign nmi_n=1'b1;
assign busrq_n=1'b1;
assign cen=1'b1;		// Clock enable
wire cclk;
assign cclk=clk|vrd;
reg [3:0]resc=4'b0010;
always @(posedge hsyn) if (resc!=0) resc<=resc-1;
assign reset_n=~((resc!=0) | reset);

tv80s #(.Mode(1)) Z80CPU(
  // Outputs
  m1_n, mreq_n, iorq_n, rd_n, wr_n, rfsh_n, halt_n, busak_n, ca, cdo,
  // Inputs
  reset_n, cclk, wait_n, int_n, nmi_n, busrq_n, cdi, cen
  );
  
assign cdi=(~mreq_n) ? mdo : iodo;

wire [7:0]mdo;	// Memory data output
wire [12:0]ma;	// memory address
wire mwe;		// memory Write enable
assign mwe=(~vrd)&(~mreq_n)&(~wr_n);
assign ma=vrd ? va : ca[12:0];

genram #(
`ifdef SIMULATION
	.INITFILE("romTAP.hex"),
`else
	.INITFILE("rand.hex"),
`endif
	.AW(13),	// 8KB
	.DW(8)
	) ram (.clk(~clk), .we(mwe), .addr(ma),.data_out(mdo),.data_in(cdo));

////////////////////////////////////////////////////////////////////////
////////////////////////////// Video ///////////////////////////////////
// External hsyn, vsyn
// Text mode: 32x24, 128 chars
// Bit MSB of char: Remark (change colors or something)
////////////////////////////////////////////////////////////////////////
reg [9:0]hc=0;  // Horizontal counter
reg hblk=0;
always @(posedge clk) begin
        if (~hsyn) hc<=10'd912; else hc<=hc+1;
        if (hc==10'd511) hblk<=1;
        else if (hc==10'd1023) hblk<=0;
end
reg [9:0]vc=990;  // Vertical Counter
reg vblk=0;
always @(negedge hblk) begin
//        if (~vsyn) vc<=10'd942; else vc<=vc+1;
        if (~vsyn) vc<=10'd990; else vc<=vc+1;
        if (vc==10'd383) vblk<=1;
        else if (vc==10'd1023) vblk<=0;
end
wire blk;	// Video blank signal
assign blk=hblk|vblk;

// Video address, shifting, & related
wire vrd;	// Read video memory
wire [12:0]va;	// video address
wire load;		// Load shift register
wire shift;		// Shift pixels
reg [7:0]char;	// ASCII value
reg [7:0]sh;	// shift register
reg remark;		// video remark
reg opaque;		// Opaque/transp
reg dblk;		// Delayed blank 

assign vrd=(hc[3:1]==3'b000)&(~blk);
assign load=vrd&hc[0]; //(hc[3:0]==4'b0001)&(~blk);
assign shift=hc[0]; //&(~blk);
always @(posedge clk) begin
	if (vrd&(~load)) char<=mdo;
	if (load) sh<=mdo; else if (shift) sh<={sh[6:0],1'bx};
	if (load) remark<=char[7];
	if (load) opaque<=char[6]|char[5]; // ASCII<32 => Transparency
	if (hc[0]) dblk<=blk;
end
assign va=load ? {3'b000,char[6:0],vc[3:1]} 	// Character pixels
			   : {3'b111,vc[8:4],hc[8:4]};		// Text in RAM


assign video=sh[7]&(~dblk);
assign rem=remark&(~dblk);
assign viden=(~dblk)&opaque;

/////////////////////////////////////////////////////
/////////////////// PERIPHERALS /////////////////////
/////////////////////////////////////////////////////
reg [7:0]iodo;	// Peripherals data output (not a reg)
wire wcontrol;	// Write to control reg
wire wspi;		// Write to SPI data
wire wrleo;		// Write to RLE timer
wire rdkey;		// Strobe on keyboard read
assign wcontrol=(~iorq_n)&(~wr_n)&(~vrd)&(~ca[1])&(~ca[0]);
assign wspi=	(~iorq_n)&(~wr_n)&(~vrd)&(~ca[1])&( ca[0]);
assign rdkey=   (~iorq_n)&(~rd_n)&(~vrd)&( ca[1])&(~ca[0]);
assign wrleo=	(~iorq_n)&(~wr_n)&(~vrd)&( ca[1])&( ca[0]);

reg [4:0]control=4'hf;	// control I/O reg
reg [7:0]rlebufo=0;		// RLE output buffer
reg [7:0]rlebufi=0;		// RLE input buffer 

// Output ports
always @(posedge clk) begin
	if(wcontrol) control<=cdo[7:4];
	if(wrleo) rlebufo<=cdo;
end
wire spifast;
assign cs1b=control[1];
assign cs2b=control[0];
assign spifast=control[2];
assign grab=control[3];

// Input ports
always @*
case (ca[1:0])
	2'b00:	iodo<={control[3:0],1'b0,motor,sbusy,kval};
	2'b01:  iodo<=ssh;
	2'b10:	iodo<=keysc;
	2'b11: 	iodo<=rlebufi;
endcase

////////////////////////////////////////////
//////////////////// SPI ///////////////////
////////////////////////////////////////////
reg [7:0]sdiv=0;
always @(negedge clk) sdiv<=wspi ? 0 : sdiv+1;
reg [7:0]ssh;
reg [7:0]shbusy=0;
reg sin;
wire spicke;
wire sbusy;
assign spicke=(sdiv==8'hff)|spifast;
assign sbusy=shbusy[7];
always @(negedge clk) begin
	if (wspi) begin ssh<=cdo; shbusy<=8'hff; end
	else if (spicke&sbusy) begin
		ssh<={ssh[6:0],smiso};
		shbusy<={shbusy[6:0],1'b0};
		end
end
assign sck=spifast? (sbusy&clk) : (sbusy&sdiv[7]);
assign mosi=ssh[7];

reg smiso;
always @(posedge sck) smiso<=miso;


//////////////////// KEYBOARD //////////////////////
// PS2 serial receiver
////////////////////////////////////////////////////

reg [10:0]kreg=11'h7FF;	// shift register, 11-bit
reg krck=1;				// sampled keyboard clock
always @(posedge clk) krck<=kclk;	

reg kock=1;			// old clock sample (for edge detection)
wire newsc;			// new scancode received
wire [7:0]scancode;	
reg [7:0] keysc;
reg kval=0;			// key valid

assign newsc=~kreg[0];		// zero start bit => scancode received
assign scancode=kreg[8:1];

// shit reg.gets filled with ones after a key is received
always @(posedge clk) begin
	kock<=krck;
	kreg<=newsc ?11'h7FF : ( ((~krck)&kock) ? {kdat,kreg[10:1]} : kreg);
	if (newsc) keysc<=scancode;
	kval<=newsc? 1 : (rdkey? 0: kval);
end

//////////////////// TIMER //////////////////////
// 
////////////////////////////////////////////////////

reg ock44k=0;	// for edge detect
reg tirq=0;		// Interrupt
reg tapeout=0;
reg [6:0]timer=0;
reg [1:0]oti=0;		// for edge detect
always @(posedge clk) begin 
	ock44k<=ck44k;
	oti<={oti[0],tapein};
end
wire tload;		// reload timer
assign tload = ((timer==0) & (ck44k & (~ock44k)))| (oti[1]^oti[0]);

always @(posedge clk) begin
	if (tload) begin 
		rlebufi<={oti[1],~timer};
		timer<=rlebufo[6:0];
		tapeout<=rlebufo[7];
		tirq<=1;
	end
	else if (ck44k & (~ock44k)) timer<=timer-1;
	if ((~iorq_n)&(~m1_n)) tirq<=0;
end
assign int_n=~tirq;

endmodule



//----------------------------------------------------------------------------
//-- Generic synchronous ROM memory
//----------------------------------------------------------------------------

module genram (        
    input clk,	// data out on rising edges
    input we,	// Write enable                  
    input wire [AW-1: 0] addr,      
    input wire [DW-1: 0] data_in,
    output reg [DW-1: 0] data_out   
);

parameter INITFILE = "rand.hex";
parameter AW = 13;
parameter DW = 8;

reg [DW-1: 0] mem [0: (1<<AW)-1];

// synchronous read
always @(posedge clk) begin
	data_out <= mem[addr];
end
// write
always @(posedge clk) begin
	if (we) mem[addr]<=data_in;
end

//-- Load contents with hex file
initial begin
  $readmemh(INITFILE, mem);
end

endmodule

